<?php
/**
 * Register trigger listeners callback.
 */
namespace Directorist\Integrations\GamiPress;

use Directorist\Review\Comment_Meta;

if ( ! defined( 'ABSPATH' ) ) {
	die();
}

/**
 * Listeners class.
 */
class Listeners {

	public static function init() {
		add_action( 'trashed_post', array( __CLASS__, 'delete_listing_listener' ) );
		add_action( 'before_delete_post', array( __CLASS__, 'delete_listing_listener' ) );
		add_action( 'transition_post_status', array( __CLASS__, 'transition_post_status_listener' ), 10, 3  );

		add_action( 'directorist_listing_reported', array( __CLASS__, 'listing_report_listener' ) );
		add_action( 'directorist_user_favorites_added', array( __CLASS__, 'user_favorites_listener' ), 10, 3 );

		add_action( 'directorist_listing_views_count_updated', array( __CLASS__, 'listing_views_count_updated_listener' ) );
		add_action( 'comment_approved_', array( __CLASS__, 'review_listener' ), 10, 2 );
		add_action( 'comment_approved_review', array( __CLASS__, 'review_listener' ), 10, 2 );
		add_action( 'directorist_review_updated', array( __CLASS__, 'review_listener' ), 10, 2 );

		add_action( 'atbdp_order_created', array( __CLASS__, 'order_created_listener' ) );
	}

	/**
	 * When a user purchases a plan, we want to trigger an event.
	 * 
	 * @param int $order_id The ID of the order.
	 */
	public static function order_created_listener( $order_id ) {
		$order   = get_post( $order_id );
		$plan_id = get_post_meta( $order_id, '_fm_plan_ordered', true );

		gamipress_trigger_event( array(
			'event'      => 'directorist_purchase_plan',
			'plan_id'    => $plan_id,
			'user_id'    => $order->post_author,
			'order_id'   => $order_id,
		) );
	}

	/**
	 * When a listing is updated, check if it's views count has changed. If it has, trigger
	 * listing_become_popular()
	 * 
	 * @param listing_id The ID of the listing that was updated.
	 */
	public static function listing_views_count_updated_listener( $listing_id ) {
		if ( ! Utils::is_listing( $listing_id ) ) {
			return;
		}
		
		self::trigger_listing_become_popular( $listing_id );
	}

	/**
	 * Listen to review activity.
	 *
	 * @param  int  $comment_id
	 * @param  mixed $comment
	 *
	 * @return void
	 */
	public static function review_listener( $comment_id, $comment ) {
		if ( is_object( $comment ) ) {
			$comment = get_object_vars( $comment );
		}
		
		$comment_id = absint( $comment_id );
		$user_id    = absint( $comment['user_id'] );
		$listing_id = absint( $comment['comment_post_ID'] );

		if ( ! Utils::is_listing( $listing_id ) ) {
			return;
		}

		// Check if review
		if ( $comment['comment_type'] !== 'review' ) {
			return;
		}

		// Check if review is approved
		if ( 1 !== (int) $comment[ 'comment_approved' ] ) {
			return;
		}

		$listing = get_post( $listing_id );
		$rating  = Comment_Meta::get_rating( $comment_id );

		self::trigger_listing_become_popular( $listing_id );

		gamipress_trigger_event( array(
			'event'      => 'directorist_new_review',
			'post_id'    => $listing_id,
			'user_id'    => $user_id,
			'comment_id' => $comment_id,
			'rating'     => $rating,
		) );

		if ( absint( $listing->post_author ) !== 0 ) {
			gamipress_trigger_event( array(
				'event'      => 'directorist_get_review',
				'post_id'    => $listing_id,
				'user_id'    => absint( $listing->post_author ),
				'comment_id' => $comment_id,
				'rating'     => $rating,
			) );
		}
	}

	/**
	 * When a listing becomes popular, trigger the "become popular listing" event
	 * 
	 * @param listing_id The ID of the listing that has become popular.
	 */
	public static function trigger_listing_become_popular( $listing_id ) {
		if ( \Directorist\Helper::is_popular( $listing_id ) ) {
			$listing           = get_post( $listing_id );
			$user_id           = (int) $listing->post_author;
			$trigger           = 'directorist_listing_become_popular';
			$already_triggered = gamipress_get_user_trigger_count( $user_id, $trigger, 0, 0, array( 'post_id' => $listing_id ) );
			
			if ( ! $already_triggered ) {
				gamipress_trigger_event( array(
					'event'   => $trigger,
					'post_id' => $listing->ID,
					'user_id' => $user_id,
				) );
			}
		}
	}

	/**
	 * Listener for listing publishing
	 *
	 * Triggers: directorist_publish_listing
	 * Triggers: directorist_listing_expired
	 *
	 * @param  string   $new_status The new post status
	 * @param  string   $old_status The old post status
	 * @param  WP_Post  $post       The post
	 *
	 * @return void
	 */
	public static function transition_post_status_listener( $new_status, $old_status, $post ) {
		if ( ! Utils::is_listing( $post ) ) {
			return;
		}

		// Statuses to check the old status
		$publish_old_statuses = array( 'new', 'auto-draft', 'draft', 'private', 'pending', 'future' );

		// Statuses to check the new status
		$publish_new_statuses = array( 'publish' );

		// Check if post status transition come to publish
		if ( in_array( $old_status, $publish_old_statuses, true ) && in_array( $new_status, $publish_new_statuses, true ) ) {

			// Trigger directorist_publish_listing
			gamipress_trigger_event( array(
				'event'     => 'directorist_publish_listing',
				'post_id'   => $post->ID,
				'user_id'   => (int) $post->post_author,
			) );
		}

		// Statuses to check the old status
		$expired_old_statuses = array( 'new', 'auto-draft', 'draft', 'publish', 'pending', 'future' );

		// Statuses to check the new status
		$expired_new_statuses = array( 'private' );

		// Check if post status transition come to publish
		if ( in_array( $old_status, $expired_old_statuses, true ) &&
			in_array( $new_status, $expired_new_statuses, true ) &&
			(string) get_post_meta( $post->ID, '_listing_status', true ) === 'expired' ) {
			// Trigger directorist_listing_expired
			gamipress_trigger_event( array(
				'event'     => 'directorist_listing_expired',
				'post_id'   => $post->ID,
				'user_id'   => $post->post_author,
			) );
		}
	}

	/**
	 * Listener for listing deletion.
	 * Triggers: directorist_delete_listing
	 *
	 * @param  integer  $post_id The deleted post ID
	 *
	 * @return void
	 */
	public static function delete_listing_listener( $listing_id ) {
		if ( ! Utils::is_listing( $listing_id ) ) {
			return;
		}

		// Return if the listing is not deleted by listing owner
		if ( get_current_user_id() !== absint( get_post_field( 'post_author', $listing_id ) ) ) {
			return;
		}

		// Trigger listing deletion actions
		gamipress_trigger_event( array(
			'event'     => 'directorist_delete_listing',
			'post_id'   => $listing_id,
			'user_id'   => absint( get_post_field( 'post_author', $listing_id ) ),
		) );
	}

	public static function listing_report_listener( $listing_id ) {
		if ( ! Utils::is_listing( $listing_id ) ) {
			return;
		}

		gamipress_trigger_event( array(
			'event'   => 'directorist_new_report',
			'post_id' => $listing_id,
			'user_id' => get_current_user_id(),
		) );

		gamipress_trigger_event( array(
			'event'   => 'directorist_get_report',
			'post_id' => $listing_id,
			'user_id' => absint( get_post_field( 'post_author', $listing_id ) ),
		) );
	}

	public static function user_favorites_listener( $user_id, $new_favorites, $old_favorites ) {
		$new_favorite_listing = array_values( array_diff( $new_favorites, $old_favorites ) );

		if ( count( $new_favorite_listing ) > 0 ) {
			gamipress_trigger_event( array(
				'event'     => 'directorist_make_favorite',
				'post_id'   => $new_favorite_listing[0],
				'user_id'   => $user_id,
			) );

			gamipress_trigger_event( array(
				'event'     => 'directorist_get_favorite',
				'post_id'   => $new_favorite_listing[0],
				'user_id'   => absint( get_post_field( 'post_author', $new_favorite_listing[0] ) ),
			) );
		}
	}
}

Listeners::init();
