<?php
/**
 * Discount coupon manager.
 */
namespace Directorist\Integrations\GamiPress;

use Exception;

if ( ! defined( 'ABSPATH' ) ) {
	die();
}

class Coupon_Manager {

	/**
	 * Ajax action name.
	 */
	const ACTION = 'directorist_gamipress_claim_discount';

	public static function init() {
		add_action( 'wp_ajax_' . self::ACTION, array( __CLASS__, 'process_claim_discount' ) );
		add_action( 'atbdp_order_created', array( __CLASS__, 'update_coupon_stats' ) );

		add_action( 'trashed_post', array( __CLASS__, 'clean_user_coupon' ) );
		add_action( 'before_delete_post', array( __CLASS__, 'clean_user_coupon' ) );
	}

	public static function clean_user_coupon( $post_id ) {
		if ( ! Utils::is_coupon_extension_enabled() ) {
			return;
		}
		
		$post = get_post( $post_id );
		if ( $post->post_type !== SWBDPC_POST_TYPE ) {
			return;
		}

		delete_metadata( 'user', null, 'directorist_gamipress_coupon_id', $post_id, true );
	}

	public static function update_coupon_stats() {
		$coupon_id = ! empty( $_REQUEST['swbdpc_coup_id'] ) ? absint( $_REQUEST['swbdpc_coup_id'] ) : 0;
		$coupon    = get_post( $coupon_id );

		if ( ! Utils::is_coupon_extension_enabled() ) {
			return;
		}

		if ( ! $coupon || $coupon->post_type !== SWBDPC_POST_TYPE ) {
			return;
		}

		$required_points = (int) get_directorist_option( 'gamipress_discount_points_required', 0 );
		$points_used     = (int) get_user_meta( get_current_user_id(), 'directorist_gamipress_used_points', true );
		$points_used     = $points_used + $required_points;

		update_user_meta( get_current_user_id(), 'directorist_gamipress_used_points', $points_used );
		delete_user_meta( get_current_user_id(), 'directorist_gamipress_coupon_id' );
	}

	public static function process_claim_discount() {
		$nonce = ! empty( $_GET['nonce'] ) ? wp_unslash( $_GET['nonce'] ) : '';
		
		try {
			if ( ! wp_verify_nonce( $nonce, self::ACTION ) ) {
				throw new Exception( 'Invalid request', 'directorist-gamipress-integration' );
			}

			if ( ! Utils::is_coupon_extension_enabled() ) {
				throw new Exception( 'Coupon extension is missing.', 'directorist-gamipress-integration' );
			}

			if ( ! Utils::is_discount_coupon_enabled() ) {
				throw new Exception( 'GamiPress discount coupon feature is disabled.', 'directorist-gamipress-integration' );
			}

			$coupon_id = get_user_meta( get_current_user_id(), 'directorist_gamipress_coupon_id', true );
			if ( $coupon_id && get_post_type( $coupon_id ) === SWBDPC_POST_TYPE ) {
				wp_send_json_success( array(
					'coupon_code' => get_post_meta( $coupon_id, 'swbdpc_coupon_code', true ),
					'message'     => __( 'Use this coupon code on checkout and enjoy discount', 'directorist-gamipress-integration' )
				) );
			}

			$total_points     = Utils::get_total_points( array( 'current_user' =>  'yes' ) );
			$required_points  = (int) get_directorist_option( 'gamipress_discount_points_required', 0 );
			$points_used      = (int) get_user_meta( get_current_user_id(), 'directorist_gamipress_used_points', true );
			$available_points = ( $total_points - $points_used );
			
			if ( $available_points < $required_points ) {
				// translators: 1: Remain points to become eligible.
				throw new Exception( sprintf(
					__( 'You are not eligible yet, you need %1$s more point(s).', 'directorist-gamipress-integration' ),
					( $required_points - $available_points )
				) );
			}

			$user_display_name = wp_get_current_user()->display_name;
			$coupon_code       = "GamiPressCoupon{$available_points}{$points_used}" . get_current_user_id();

			$coupon_id = wp_insert_post( array(
				'post_title'   => sprintf( 'GamiPress discount coupon [%s]', $user_display_name ),
				'post_type'    => SWBDPC_POST_TYPE,
				'post_status'  => 'publish',
				'post_excerpt' => sprintf( 'GamiPress discount coupon generated for %s (#%s)', $user_display_name, get_current_user_id() ),
				'meta_input'   => array(
					'swbdpc_coupon_descrip'     => sprintf( 'GamiPress discount coupon generated for %s (#%s)', $user_display_name, get_current_user_id() ),
					'swbdpc_coupon_code'        => $coupon_code,
					'swbdpc_coupon_type'        => self::get_discount_type(),
					'swbdpc_coupon_amount'      => get_directorist_option( 'gamipress_discount_amount', 0 ),
					'swbdpc_coupon_usage_limit' => 1,
				)
			) );

			if ( is_wp_error( $coupon_id ) || ! $coupon_id ) {
				throw new Exception( __( 'Something went wrong, please try again.', 'directorist-gamipress-integration' ) );
			}
			
			update_user_meta( get_current_user_id(), 'directorist_gamipress_coupon_id', $coupon_id );

			wp_send_json_success( sprintf( esc_html__( 'Coupon code: %s' ) , '<code>' . $coupon_code . '</code>' ) );
		} catch( Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}
	}

	public static function get_discount_type() {
		$map = array(
			'percent-discount' => 'percentage_dis',
			'fixed-discount'   => 'fixed_cart_dis',
		);

		$discount_type = get_directorist_option( 'gamipress_discount_type', 'fixed-discount' );

		return ( isset( $map[ $discount_type ] ) ? $map[$discount_type ] : $map['fixed-discount'] );
	}
}

Coupon_Manager::init();
