<?php
/**
 * BuddyPress Group Listings Component.
 * 
 * @package wpWax\Directorist\BuddyPress
 * @since Directorist\BuddyPress 1.0.0
 */
namespace wpWax\Directorist\BuddyPress;

defined( 'ABSPATH' ) || die();

use BP_Group_Extension;

/**
 * Loads Group Extension for Listings Component
 */
class Group_Listings_Extension extends BP_Group_Extension {

	const AJAX_ACTION = 'dbp_get_listings_ids';

	/** Methods ***************************************************************/

	/**
	 * Setup Listings group extension variables
	 */
	public function __construct() {
		$this->setup_variables();
		$this->setup_actions();
		$this->maybe_unset_listings_menu();
	}

	/**
	 * Setup the group forums class variables
	 */
	private function setup_variables() {

		// Component Name
		$this->name          = __( 'Listings', 'directorist-buddypress-integration' );
		$this->nav_item_name = __( 'Listings', 'directorist-buddypress-integration' );

		// Slug used in URL
		$this->slug = 'listings';

		// Listings component is visible
		$this->visibility = 'public';

		// Set positions towards end
		$this->create_step_position = 15;
		$this->nav_item_position    = 10;

		// Allow create step and show in nav
		$this->enable_create_step = true;
		$this->enable_nav_item    = true;
		$this->enable_edit_item   = true;

		// Template file to load, and action to hook display on to
		$this->template_file = 'groups/single/plugins';
		$this->display_hook  = 'bp_template_content';
	}

	/**
	 * Setup the group forums class actions
	 */
	private function setup_actions() {

		// Forums needs to listen to BuddyPress group deletion.
		add_action( 'groups_before_delete_group', array( $this, 'disconnect_listings_from_group' ) );

		// Adds a Listings metabox to the new BuddyPress Group Admin UI
		add_action( 'bp_groups_admin_meta_boxes', array( $this, 'group_admin_ui_edit_screen' ) );

		// Saves the Listings options if they come from the BuddyPress Group Admin UI
		add_action( 'bp_group_admin_edit_after', array( $this, 'edit_screen_save' ) );

		// Enqueue assets
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_enqueue_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );

		add_action( 'wp_ajax_' . self::AJAX_ACTION, array( $this, 'get_listings_ids' ) );

		add_filter( 'bp_nouveau_nav_has_count', function( $has_count, $nav_item, $displayed_nav ) {
			if ( $displayed_nav === 'groups' && $nav_item->slug === 'listings' && $nav_item->parent_slug === groups_get_current_group()->slug ) {
				return true;
			}

			return $has_count;
		}, 10, 3 );

		add_filter( 'bp_nouveau_get_nav_count', function( $count, $nav_item, $displayed_nav ) {
			if ( $displayed_nav !== 'groups' || $nav_item->slug !== 'listings' ) {
				return $count;
			}

			$listings_ids = dbp_get_group_connected_listings_ids( groups_get_current_group()->id );
			if ( ! empty( $listings_ids ) && is_array( $listings_ids ) ) {
				$count = count( $listings_ids );
			}

			return $count;
		}, 10, 3 );
	}

	public function get_listings_ids() {
		$nonce       = isset( $_REQUEST['nonce'] ) ? wp_unslash( $_REQUEST['nonce'] ) : '';
		$search_term = isset( $_REQUEST['search'] ) ? sanitize_text_field( trim( $_REQUEST['search'] ) ) : '';

		try {
			if ( ! wp_verify_nonce( $nonce, self::AJAX_ACTION ) ) {
				throw new \Exception( __( 'Invalid request', 'directorist-buddypress-integration' ), 400 );
			}

			if ( empty( $search_term ) ) {
				$data = array(
					'id'   => -1,
					'text' => __( 'Nothing found, try different search term.', 'directorist-buddypress-integration' ),
				);

				wp_send_json_success( $data );
			}

			$args = array(
				'post_type'              => ATBDP_POST_TYPE,
				'post_status'            => 'publish',
				'orderby'                => 'relevance',
				'posts_per_page'         => 10,
				's'                      => $search_term,
				'no_found_rows'          => true,
				'update_post_meta_cache' => false,
				'update_post_term_cache' => false,
			);

			$query = new \WP_Query( $args );
			$data = array();

			if ( $query->have_posts() ) {
				$data = array_map( function( $post ) { 
					return array(
						'id'   => $post->ID,
						'text' => strip_tags( $post->post_title ),
					);
				}, $query->posts );
			}

			wp_send_json_success( $data );
		} catch( \Exception $e ) {
			wp_send_json_error( $e->getMessage() );
		}
	}

	/**
	 * Delete connected group listings ids.
	 *
	 * @param int $group_id
	 *
	 * @return void
	 */
	public function disconnect_listings_from_group( $group_id ) {
		dbp_delete_group_connected_listings_ids( $group_id );
	}

	public function frontend_enqueue_scripts() {
		if ( bp_is_current_action( 'admin' ) && bp_is_action_variable( $this->slug ) ) {
			$this->enqueue_scripts();
		}
	}

	public function admin_enqueue_scripts() {
		if ( isset( $_GET['page'] ) && $_GET['page'] === 'bp-groups' ) {
			$this->enqueue_scripts();
		}
	}

	protected function enqueue_scripts() {
		if ( ! ( wp_style_is( 'bp-select2' ) && wp_style_is( 'directorist-select2' ) ) ) {
			wp_enqueue_style(
				'select2',
				DIRECTORIST_VENDOR_CSS . 'select2.min.css',
				directorist_buddypress()->version,
				true
			);
		}

		if ( ! ( wp_script_is( 'bp-select2' ) && wp_script_is( 'directorist-select2' ) ) ) {
			wp_enqueue_script(
				'jquery-select2',
				DIRECTORIST_VENDOR_JS . 'select2.min.js',
				array( 'jquery' ),
				directorist_buddypress()->version,
				true
			);
		}

		wp_enqueue_script(
			'directorist-buddypress-integration',
			directorist_buddypress()->plugin_url . 'assets/js/main.js',
			array( 'jquery' ),
			directorist_buddypress()->version,
			true
		);

		wp_localize_script(
			'directorist-buddypress-integration',
			'DBB',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( self::AJAX_ACTION ),
				'action'  => self::AJAX_ACTION,
			)
		);
	}

	/**
	 * The primary display function for group forums
	 *
	 * @param int $group_id ID of the current group. Available only on BP 2.2+.
	 */
	public function display( $group_id = null ) {
		$listings_ids = dbp_get_group_connected_listings_ids( $group_id );
		
		if ( empty( $listings_ids ) ) : ?>

			<p class="directorist-group-listings-not-found"><em><?php esc_html_e( 'Group does not have any listings yet.', 'directorist-buddypress-integration' ); ?></em></p>
		
		<?php else :

			$group = groups_get_group( $group_id );
			$args = array(
				'orderby'             => 'date',
				'order'               => 'DESC',
				'view'                => get_directorist_option( 'bp_group_listings_view', 'grid' ),
				'advanced_filter'     => 'no',
				'logged_in_user_only' => $group->user_has_access ? 'no' : 'yes',
				'listings_per_page'   => get_directorist_option( 'bp_group_listings_per_page', 6 ),
				'ids'                 => implode( ',', $listings_ids ),
			);
			
			echo dbp_run_shortcode_callback( 'directorist_all_listing', $args );
		
		endif;

		// Hide breadcrumb
		add_filter( 'bbp_no_breadcrumb', '__return_true' );
	}

	/**
	 * Maybe unset the group forum nav item if group does not have a forum
	 *
	 * @return If not viewing a single group
	 */
	public function maybe_unset_listings_menu() {
		// Bail if not viewing a single group
		if ( ! bp_is_group() ) {
			return;
		}

		// Tweak the nav item variable based on if group has forum or not
		$this->enable_nav_item = (bool) dbp_get_new_group_enable_listings();
	}

	/** Edit ******************************************************************/

	/**
	 * Show forums and new forum form when editing a group
	 * 
	 * @param object $group (the group to edit if in Group Admin UI)
	 * @uses is_admin() To check if we're in the Group Admin UI
	 */
	public function edit_screen( $group = false ) {
		$group_id  = empty( $group->id ) ? bp_get_new_group_id() : $group->id;

		if ( is_admin() ) {
			$listings_ids = dbp_get_group_connected_listings_ids( $group_id );
		} else {
			$listings_ids = dbp_get_group_connected_listings_ids( bp_get_group_id() );
		}

		// Should box be checked already?
		$checked = is_admin() ? dbp_is_group_listings_enabled( $group_id ) : dbp_is_group_listings_enabled( bp_get_group_id() ); ?>

		<h4 class="bb-section-title"><?php esc_html_e( 'Group Listings Settings', 'directorist-buddypress-integration' ); ?></h4>

		<fieldset>
			<legend class="screen-reader-text"><?php esc_html_e( 'Group Listings Settings', 'directorist-buddypress-integration' ); ?></legend>
			<!-- <p class="bb-section-info"><?php esc_html_e( 'Connect one or more directory listings to allow members of this group to ........', 'directorist-buddypress-integration' ); ?></p> -->

			<div class="field-group">
				<p class="checkbox bp-checkbox-wrap bp-group-option-enable">
					<input type="checkbox" name="dbp_edit_enable_group_listings" id="dbp_edit_enable_group_listings" class="bs-styled-checkbox" value="1"<?php checked( $checked ); ?> />
					<label for="dbp_edit_enable_group_listings"><?php esc_html_e( 'Yes, I want this group to have a directory listings.', 'directorist-buddypress-integration' ); ?></label>
				</p>
			</div>

			<hr class="bb-sep-line" />
			<div class="field-group">
				<h4 class="bb-section-title"><?php esc_html_e( 'Connected Listings', 'directorist-buddypress-integration' ); ?></h4>
				<p class="bb-section-info"><?php esc_html_e( 'Connect as many listings as you want with this group. Just search, select and save.', 'directorist-buddypress-integration' ); ?></p>
				<?php
				dbp_listings_dropdown( array(
					'class' => 'directorist-listings-dropdown',
					'id'    => 'directorist-listings-dropdown',
					'value' => $listings_ids,
				) );
				?>
			</div>

			<?php if ( ! is_admin() ) : ?>
				<br />
				<input type="submit" value="<?php esc_attr_e( 'Save Settings', 'directorist-buddypress-integration' ); ?>" />
			<?php endif; ?>

		</fieldset>

		<?php

		// Verify intent
		if ( is_admin() ) {
			wp_nonce_field( 'groups_edit_save_' . $this->slug, 'listings_group_admin_ui' );
		} else {
			wp_nonce_field( 'groups_edit_save_' . $this->slug );
		}
	}

	/**
	 * Save the Group Forum data on edit
	 * 
	 * @param int $group_id (to handle Group Admin UI hook bp_group_admin_edit_after )
	 */
	public function edit_screen_save( $group_id = 0 ) {
		// Bail if not a POST action
		if ( 'POST' !== strtoupper( $_SERVER['REQUEST_METHOD'] ) ) {
			return;
		}

		// Admin Nonce check
		if ( is_admin() ) {
			check_admin_referer( 'groups_edit_save_' . $this->slug, 'listings_group_admin_ui' );

			// Theme-side Nonce check
		} elseif ( ! isset( $_POST['_wpnonce'] ) || ( isset( $_POST['_wpnonce'] ) && ! wp_verify_nonce( $_POST['_wpnonce'], 'groups_edit_save_' . $this->slug ) ) ) {
			return;
		}

		$edit_listings = ! empty( $_POST['dbp_edit_enable_group_listings'] ) ? true : false;
		$group_id   = ! empty( $group_id ) ? $group_id : bp_get_current_group_id();

		$listings = ! empty( $_POST['directorist_listings_dropdown'] ) ? array_map( 'absint', (array) $_POST['directorist_listings_dropdown'] ) : array();

		// Update the group listings settings
		$this->toggle_group_listings( $group_id, $edit_listings );

		// Update group connected listings
		if ( ! empty( $listings ) ) {
			dbp_update_group_connected_listings_ids( $group_id, $listings );
		} else {
			dbp_delete_group_connected_listings_ids( $group_id );
		}

		/**
		 * Add action that fire before user redirect
		 *
		 * @param int $group_id Current group id
		 */
		do_action( 'bp_group_admin_after_edit_screen_save', $group_id );

		// Redirect after save when not in admin
		if ( ! is_admin() ) {
			bp_core_redirect( trailingslashit( bp_get_group_permalink( buddypress()->groups->current_group ) . '/admin/' . $this->slug ) );
		}
	}

	/**
	 * Adds a metabox to BuddyPress Group Admin UI
	 *
	 * @uses add_meta_box
	 * @uses BBP_Forums_Group_Extension::group_admin_ui_display_metabox() To display the edit screen
	 */
	public function group_admin_ui_edit_screen() {
		add_meta_box(
			'directorist_group_admin_ui_meta_box',
			__( 'Directory Listings', 'directorist-buddypress-integration' ),
			array( $this, 'group_admin_ui_display_metabox' ),
			get_current_screen()->id,
			'side',
			'core'
		);
	}

	/**
	 * Displays the Forums metabox in BuddyPress Group Admin UI
	 *
	 * @param object $item (group object)
	 * @uses add_meta_box
	 * @uses BBP_Forums_Group_Extension::edit_screen() To get the html
	 */
	public function group_admin_ui_display_metabox( $item ) {
		$this->edit_screen( $item );
	}

	/** Create ****************************************************************/

	/**
	 * Show forums and new forum form when creating a group
	 *
	 * @since bbPress (r3465)
	 */
	public function create_screen( $group_id = 0 ) {

		// Bail if not looking at this screen
		if ( ! bp_is_group_creation_step( $this->slug ) ) {
			return false;
		}

		// Check for possibly empty group_id
		if ( empty( $group_id ) ) {
			$group_id = bp_get_new_group_id();
		}

		$checked = bp_get_new_group_enable_forum() || groups_get_groupmeta( $group_id, 'forum_id' );

		wp_nonce_field( 'dbp_create_enable_group_listings', '_dbp_enable_group_nonce' );
		?>

		<h4><?php esc_html_e( 'Group Directory Listings', 'directorist-buddypress-integration' ); ?></h4>

		<p><?php esc_html_e( 'Create a discussion forum to allow members of this group to communicate in a structured, bulletin-board style fashion.', 'directorist-buddypress-integration' ); ?></p>

		<p class="checkbox bp-checkbox-wrap">
			<input type="checkbox" name="dbp_create_enable_group_listings" id="dbp_create_enable_group_listings" class="bs-styled-checkbox" value="1"<?php checked( $checked ); ?> />
			<label for="dbp_create_enable_group_listings"><?php esc_html_e( 'Yes, I want this group to have a directory listings.', 'directorist-buddypress-integration' ); ?></label>
		</p>
		<?php
	}

	/**
	 * Save the Group Forum data on create
	 *
	 * @since bbPress (r3465)
	 */
	public function create_screen_save( $group_id = 0 ) {
		$nonce = isset( $_POST['_dbp_enable_group_nonce'] ) ? $_POST['_dbp_enable_group_nonce'] : '';

		//Nonce check
		if ( ! wp_verify_nonce( $nonce, 'dbp_create_enable_group_listings' ) ) {
			return;
		}

		// Check for possibly empty group_id
		if ( empty( $group_id ) ) {
			$group_id = bp_get_new_group_id();
		}

		$enabled_listings = ! empty( $_POST['dbp_create_enable_group_listings'] ) ? true : false;

		// Toggle listings on
		$this->toggle_group_listings( $group_id, $enabled_listings );
	}

	/**
	 * Toggle the directory listings group setting on or off
	 *
	 * @param int  $group_id The group to toggle
	 * @param bool $enabled True for on, false for off
	 * @uses groups_get_group() To get the group to toggle
	 * @return False if group is not found, otherwise return the group
	 */
	public function toggle_group_listings( $group_id = 0, $enabled = false ) {
		
		// Get the group
		$group = groups_get_group( array( 'group_id' => $group_id ) );

		// Bail if group cannot be found
		if ( empty( $group ) ) {
			return false;
		}

		if ( $enabled ) {
			dbp_enable_group_listings( $group_id );
		} else {
			dbp_disable_group_listings( $group_id );
		}

		// Return the group
		return $group;
	}
}
