<?php
/**
 * BuddyPress listings component.
 * 
 * @package wpWax\Directorist\BuddyPress
 * @since Directorist\BuddyPress 1.0.0
 */
namespace wpWax\Directorist\BuddyPress;

defined( 'ABSPATH' ) || die();

use BP_Component;

/**
 * Defines the Listings Component.
 */
class BP_Listings_Component extends BP_Component {

    /**
	 * Start the listings component creation process.
	 */
	public function __construct() {
		parent::start(
			'listings',
			__( 'Listings', 'directorist-buddypress-integration' ),
			directorist_buddypress()->plugin_dir . 'includes'
		);

		$this->includes();

		buddypress()->active_components[ $this->id ] = '1';
	}

	public function setup_actions() {
		// Setup the components.
		add_action( 'bp_init', array( $this, 'setup_components' ), 7 );

		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_scripts' ) );

		add_filter( 'directorist_single_listing_social_sharing_items', array( $this, 'add_share_link' ) );

		parent::setup_actions();
	}
	
	/**
	 * Add timeline share link.
	 *
	 * @param array $share_links
	 *
	 * @return array
	 */
	public function add_share_link( $share_links ) {
		if ( dbp_is_admin_listing_sharing_enabled() && is_user_logged_in() ) {
			$url = wp_nonce_url(
				trailingslashit( bp_loggedin_user_domain() . dbp_get_listings_slug() )
				. 'favorite/share/' . get_the_ID(),
				'dbp_single_listing_share_on_timeline'
			);
			
			$share_links['bp-share'] = array(
				'title' => get_directorist_option( 'bp_listing_sharing_text', __( 'On Timeline', 'directorist-buddypress-integration' ) ),
				'icon'  => 'fas fa-share-alt',
				'link'  => $url,
			);
		}

		return $share_links;
	}

	/**
	 * Enqueue assets.
	 *
	 * @return void
	 */
	public function enqueue_frontend_scripts() {
		if ( bp_is_current_component( 'groups' ) || bp_is_current_component( 'listings' ) ) {
			wp_enqueue_style(
				'directorist-buddypress-integration',
				directorist_buddypress()->plugin_url . 'assets/css/main.css',
				null,
				directorist_buddypress()->version
			);
		}
	}

	public function setup_components() {
		// Register the listings extension only if groups are active
		if ( dbp_is_admin_group_listings_enabled() && bp_is_active( 'groups' ) ) {
			
			/**
			 * need to remove this hooks before group extension because
			 * it was checking for access to that post before wp handles the post id assign
			 */
			if ( bp_is_group() ) {
				remove_action( 'bbp_template_redirect', 'bbp_check_forum_edit', 10 );
				remove_action( 'bbp_template_redirect', 'bbp_check_topic_edit', 10 );
				remove_action( 'bbp_template_redirect', 'bbp_check_reply_edit', 10 );
			}

			bp_register_group_extension( '\wpWax\Directorist\BuddyPress\Group_Listings_Extension' );
		}
	}

	/**
	 * Include bp-listings files.
	 *
	 * @see BP_Component::includes() for description of parameters.
	 *
	 * @param array $includes See {@link BP_Component::includes()}.
	 */
	public function includes( $includes = array() ) {
		$includes = array(
			'bp-listings-functions.php',
			'bp-listings-screens.php',
			'bp-listings-template.php',
			'bp-listings-actions.php',
		);

		// Conditional includes.
		if ( bp_is_active( 'activity' ) ) {
			$includes[] = 'bp-listings-activity.php';
		}

		// Include group listings extension conditionally.
		if ( dbp_is_admin_group_listings_enabled() && bp_is_active( 'groups' ) ) {
			$includes[] = 'class-group-listings-extension.php';
		}

		parent::includes( $includes );
	}

	/**
	 * Set up bp-listings global settings.
	 *
	 * @see BP_Component::setup_globals() for description of parameters.
	 *
	 * @param array $args See {@link BP_Component::setup_globals()}.
	 */
	public function setup_globals( $args = array() ) {
		$args = array(
			'slug'                  => $this->id,
			'has_directory'         => false,
			// 'notification_callback' => 'notfication_callback_function',
		);

		parent::setup_globals( $args );
	}

	/**
	 * Set up component navigation.
	 *
	 * @see BP_Component::setup_nav() for a description of arguments.
	 *
	 * @param array $main_nav Optional. See BP_Component::setup_nav() for
	 *                        description.
	 * @param array $sub_nav  Optional. See BP_Component::setup_nav() for
	 *                        description.
	 */
	public function setup_nav( $main_nav = array(), $sub_nav = array() ) {
		// Determine user to use.
		if ( bp_displayed_user_domain() ) {
			$user_domain = bp_displayed_user_domain();
		} elseif ( bp_loggedin_user_domain() ) {
			$user_domain = bp_loggedin_user_domain();
		} else {
			return;
		}

		$slug           = dbp_get_listings_slug();
		$parent_url     = trailingslashit( $user_domain . $slug );
		$current_action = bp_current_action();
		$listings_count = 0;
		
		if ( $current_action === 'favorites' ) {
			$user_favourites = get_user_meta( bp_displayed_user_id(), 'atbdp_favourites', true );
			$listings_count  = is_array( $user_favourites ) ? count( $user_favourites ) : 0;
		} else {
			$listings_count = dbp_count_user_listings( bp_displayed_user_id() );
		}

		$title = sprintf(
			__( 'Listings %s', 'directorist-buddypress-integration' ),
			'<span class="count">' . number_format( $listings_count ) . '</span>'
		);

		$main_nav = array(
			'default_subnav_slug'     => 'my-listings',
			'item_css_id'             => $this->id,
			'name'                    => $title,
			'position'                => 80,
			'screen_function'         => 'dbp_listings_screen_my_listings',
			'slug'                    => $slug,
		);

		// Add the subnav items to the listings nav item.
		$sub_nav[] = array(
			'name'            => bp_is_my_profile() ? __( 'My Listings', 'directorist-buddypress-integration' ) : __( 'Listings', 'directorist-buddypress-integration' ),
			'parent_slug'     => $slug,
			'parent_url'      => $parent_url,
			'position'        => 10,
			'screen_function' => 'dbp_listings_screen_my_listings',
			'slug'            => 'my-listings',
		);

		$sub_nav[] = array(
			'name'            => __( 'Favorites', 'directorist-buddypress-integration' ),
			'parent_slug'     => $slug,
			'parent_url'      => $parent_url,
			'position'        => 20,
			'screen_function' => 'dbp_listings_screen_favorites',
			'slug'            => 'favorites',
		);

		if ( bp_is_my_profile() ) {
			$sub_nav[] = array(
				'name'            => __( 'Add Listing', 'directorist-buddypress-integration' ),
				'parent_slug'     => $slug,
				'parent_url'      => $parent_url,
				'position'        => 30,
				'screen_function' => 'dbp_listings_screen_add_listing',
				'slug'            => 'add',
			);
		}
		
		$sub_nav = apply_filters( 'dbp_listings_screen_my_listings_subnavs', $sub_nav, $slug, $parent_url );

		parent::setup_nav( $main_nav, $sub_nav );
	}

	/**
	 * Set up listings integration with the WordPress admin bar.
	 *
	 * @see BP_Component::setup_admin_bar() for a description of arguments.
	 *
	 * @param array $wp_admin_nav See BP_Component::setup_admin_bar()
	 *                            for description.
	 */
	public function setup_admin_bar( $wp_admin_nav = array() ) {
		// Menus for logged in user.
		if ( is_user_logged_in() ) {
			// Setup the logged in user variables.
			$listings_link = trailingslashit( bp_loggedin_user_domain() . dbp_get_listings_slug() );

			// Add the "My Account" sub menus.
			$wp_admin_nav[] = array(
				'parent' => buddypress()->my_account_menu_id,
				'id'     => 'my-account-' . $this->id,
				'title'  => __( 'Listings', 'directorist-buddypress-integration' ),
				'href'   => $listings_link,
			);

			// My listings.
			$wp_admin_nav[] = array(
				'parent'   => 'my-account-' . $this->id,
				'id'       => 'my-account-' . $this->id . '-listings',
				'title'    => __( 'My Listings', 'directorist-buddypress-integration' ),
				'href'     => $listings_link,
				'position' => 10,
			);

			$wp_admin_nav[] = array(
				'parent'   => 'my-account-' . $this->id,
				'id'       => 'my-account-' . $this->id . '-favorites',
				'title'    => __( 'Favorites', 'directorist-buddypress-integration' ),
				'href'     => trailingslashit( $listings_link . 'favorites' ),
				'position' => 20,
			);

			$wp_admin_nav[] = array(
				'parent'   => 'my-account-' . $this->id,
				'id'       => 'my-account-' . $this->id . '-add',
				'title'    => __( 'Add Listing', 'directorist-buddypress-integration' ),
				'href'     => trailingslashit( $listings_link . 'add' ),
				'position' => 30,
			);
		}

		parent::setup_admin_bar( $wp_admin_nav );
	}
}
