<?php
/**
 * The -functions.php file is a good place to store miscellaneous functions needed by your plugin.
 *
 * @package wpWax\Directorist\BuddyPress
 */
defined( 'ABSPATH' ) || die();

/**
 * dbp_listings_load_template_filter()
 *
 * You can define a custom load template filter for your component. This will allow
 * you to store and load template files from your plugin directory.
 *
 * This will also allow users to override these templates in their active theme and
 * replace the ones that are stored in the plugin directory.
 *
 * If you're not interested in using template files, then you don't need this function.
 *
 * This will become clearer in the function dbp_listings_screen_one() when you want to load
 * a template file.
 */
function dbp_listings_load_template_filter( $found_template, $templates ) {
	global $bp;

	/**
	 * Only filter the template location when we're on the example component pages.
	 */
	if ( $bp->current_component != $bp->listings->slug ) {
		return $found_template;
	}

	/*
	 * $found_template is not empty when the older template files are found in the
	 * parent and child theme.
	 *
	 * /wp-content/themes/YOUR-THEME/members/single/example.php
	 *
	 * The older template files utilize a full template ( get_header() +
	 * get_footer() ), which doesn't work for themes and theme compat.
	 *
	 * When the older template files are not found, we use our new template method,
	 * which will act more like a template part.
	 */
	if ( empty( $found_template ) ) {
		/*
		 * Register our theme compat directory.
		 *
		 * This tells BP to look for templates in our plugin directory last
		 * when the template isn't found in the parent / child theme
		 */
		bp_register_template_stack( 'dbp_listings_get_template_directory', 14 );

		/*
		 * locate_template() will attempt to find the plugins.php template in the
		 * child and parent theme and return the located template when found
		 *
		 * plugins.php is the preferred template to use, since all we'd need to do is
		 * inject our content into BP.
		 *
		 * Note: this is only really relevant for bp-default themes as theme compat
		 * will kick in on its own when this template isn't found.
		 */
		$found_template = locate_template( 'members/single/plugins.php', false, false );

		// Add our hook to inject content into BP.
		add_action(
			'bp_template_content',
			function() use ( $templates ) {
				foreach ( $templates as $template ) {
					$template_name = str_replace( '.php', '', $template );

					/*
					 * Only add the template to the content when it's not the generic
					 * plugins.php template  to avoid infinite loop.
					 */
					if ( 'members/single/plugins' !== $template_name ) {
						bp_get_template_part( $template_name );
					}
				}
			}
		);
	}

	return apply_filters( 'dbp_listings_load_template_filter', $found_template );
}
add_filter( 'bp_located_template', 'dbp_listings_load_template_filter', 10, 2 );

/**
 * Get the Directorist BuddyPress template directory.
 *
 * @uses apply_filters()
 * @return string
 */
function dbp_listings_get_template_directory() {
	return apply_filters( 'dbp_listings_get_template_directory', directorist_buddypress()->plugin_dir . 'includes/templates' );
}

/**
 * Fix my-listings action page pagination issue.
 * 
 * The pagination doesn't work when my-listings is selected as default or missing from url.
 * 
 * @param string $result
 * 
 * @return string
 */
function dbp_fix_listings_default_action_pagination( $result ) {
	if ( ( bp_is_current_action( 'my-listings' ) ) && strstr( $result, '/my-listings/' ) === false ) {
		$result = str_replace( '/listings/', '/listings/my-listings/', $result );
	}
	
	return $result;
}

/**
 * Add fix listings pagination hook
 *
 * @return void
 */
function dbp_start_fix_listings_default_action() {
	add_filter( 'get_pagenum_link', 'dbp_fix_listings_default_action_pagination' );
}
add_action( 'bp_before_member_my-listings_content', 'dbp_start_fix_listings_default_action' );

/**
 * Remove fix listings pagination hook
 *
 * @return void
 */
function dbp_end_fix_listings_default_action() {
	remove_filter( 'get_pagenum_link', 'dbp_fix_listings_default_action_pagination' );
}
add_action( 'bp_after_member_my-listings_content', 'dbp_end_fix_listings_default_action' );

/**
 * Call a shortcode function by tag name.
 *
 * @param string $tag     The shortcode whose function to call.
 * @param array  $atts    The attributes to pass to the shortcode function. Optional.
 * @param array  $content The shortcode's content. Default is null (none).
 *
 * @return string|bool False on failure, the result of the shortcode on success.
 */
function dbp_run_shortcode_callback( $tag, array $atts = array(), $content = null ) {
	global $shortcode_tags;

	if ( ! isset( $shortcode_tags[ $tag ] ) ) {
		return false;
	}

	return call_user_func( $shortcode_tags[ $tag ], $atts, $content, $tag );
}

/**
 * Add extra param in all listings query arguments.
 *
 * @param array $args Query arguments.
 * 
 * @return array
 */
function dbp_update_all_listings_query_arguments( $args ) {
	if ( bp_is_current_action( 'my-listings' ) ) {
		$args['author'] = bp_displayed_user_id();
	}

	return $args;
}
add_filter( 'atbdp_all_listings_query_arguments', 'dbp_update_all_listings_query_arguments' );

/**
 * Check if directory listings is enabled for the given group.
 * 
 * @param int $group_id
 * 
 * @return bool
 */
function dbp_is_group_listings_enabled( $group_id ) {
	return (bool) groups_get_groupmeta( $group_id, 'directorist_listings_enabled' );
}

/**
 * Enable directory listings for the given group.
 * 
 * @return void
 */
function dbp_enable_group_listings( $group_id ) {
	groups_update_groupmeta( $group_id, 'directorist_listings_enabled', true );
}

/**
 * Disable directory listings for the given group.
 * 
 * @return void
 */
function dbp_disable_group_listings( $group_id ) {
	groups_delete_groupmeta( $group_id, 'directorist_listings_enabled' );
}

/**
 * Check if the newly created group has listings enabled.
 *
 * @return bool
 */
function dbp_get_new_group_enable_listings() {
	$out = isset( buddypress()->groups->current_group )
		? dbp_is_group_listings_enabled( buddypress()->groups->current_group->id )
		: false;

	return $out;
}

/**
 * Get group listings ids.
 * 
 * @param int $group_id
 * 
 * @return int[] Listings ids.
 */
function dbp_get_group_connected_listings_ids( $group_id ) {
	return groups_get_groupmeta( $group_id, 'directorist_listings_ids' );
}

/**
 * Save group connected listings.
 * 
 * @return void
 */
function dbp_update_group_connected_listings_ids( $group_id, $ids = array() ) {
	groups_update_groupmeta( $group_id, 'directorist_listings_ids', $ids );
}

/**
 * Delete group connected listings ids.
 * 
 * @return void
 */
function dbp_delete_group_connected_listings_ids( $group_id ) {
	groups_delete_groupmeta( $group_id, 'directorist_listings_ids' );
}
