<?php

/**
 * dbp_listings_record_activity()
 *
 * If the activity stream component is installed, this function will record activity items for your
 * component.
 *
 * You must pass the function an associated array of arguments:
 *
 *     $args = array(
 *       REQUIRED PARAMS
 *       'action' => For example: "Andy high-fived John", "Andy posted a new update".
 *       'type' => The type of action being carried out, for example 'new_friendship', 'joined_group'. This should be unique within your component.
 *
 *       OPTIONAL PARAMS
 *       'id' => The ID of an existing activity item that you want to update.
 *       'content' => The content of your activity, if it has any, for example a photo, update content or blog post excerpt.
 *       'component' => The slug of the component.
 *       'primary_link' => The link for the title of the item when appearing in RSS feeds (defaults to the activity permalink)
 *       'item_id' => The ID of the main piece of data being recorded, for example a group_id, user_id, forum_post_id - useful for filtering and deleting later on.
 *       'user_id' => The ID of the user that this activity is being recorded for. Pass false if it's not for a user.
 *       'recorded_time' => (optional) The time you want to set as when the activity was carried out (defaults to now)
 *       'hide_sitewide' => Should this activity item appear on the site wide stream?
 *       'secondary_item_id' => (optional) If the activity is more complex you may need a second ID. For example a group forum post may need the group_id AND the forum_post_id.
 *     )
 *
 * Example usage would be:
 *
 *   dbp_listings_record_activity( array( 'type' => 'new_highfive', 'action' => 'Andy high-fived John', 'user_id' => $bp->loggedin_user->id, 'item_id' => $bp->displayed_user->id ) );
 *
 */
function dbp_listings_record_activity( $args = '' ) {
	global $bp;

	if ( ! function_exists( 'bp_activity_add' ) ) {
		return false;
	}

	$defaults = array(
		'id'                => false,
		'user_id'           => bp_loggedin_user_id(),
		'action'            => '',
		'content'           => '',
		'primary_link'      => '',
		'component'         => $bp->listings->id,
		'type'              => false,
		'item_id'           => false,
		'secondary_item_id' => false,
		'recorded_time'     => gmdate( 'Y-m-d H:i:s' ),
		'hide_sitewide'     => false,
	);

	$r = wp_parse_args( $args, $defaults );
	extract( $r );

	return bp_activity_add(
		array(
			'id'                => $id,
			'user_id'           => $user_id,
			'action'            => $action,
			'content'           => $content,
			'primary_link'      => $primary_link,
			'component'         => $component,
			'type'              => $type,
			'item_id'           => $item_id,
			'secondary_item_id' => $secondary_item_id,
			'recorded_time'     => $recorded_time,
			'hide_sitewide'     => $hide_sitewide,
		)
	);
}


/**
 * Register the activity type.
 *
 * @since 1.1.0
 */
function dbp_listings_register_activity_actions() {

	bp_activity_set_action(
		buddypress()->listings->id,
		'new_listing',
		__( 'Published a listing', 'directorist-buddypress-integration' ),
		'dbp_format_activity_action_new_listing',
		__( 'New Listings', 'directorist-buddypress-integration' ),
		array( 'activity' )
	);

	bp_activity_set_action(
		buddypress()->listings->id,
		'shared_listing',
		__( 'Shared a listing', 'directorist-buddypress-integration' ),
		'dbp_format_activity_action_share_listing',
		__( 'Listings Shared', 'directorist-buddypress-integration' ),
		array( 'activity' )
	);

}
add_action( 'bp_register_activity_actions', 'dbp_listings_register_activity_actions' );

function dbp_format_activity_action_new_listing( $action, $activity ) {	
	$user_link = bp_core_get_userlink( $activity->user_id );

	if ( $activity->item_id ) {
		$action = sprintf(
			__( '%s added a new listing - %s', 'directorist-buddypress-integration' ),
			$user_link,
			sprintf( '<a href="%s">%s</a>', get_the_permalink( $activity->item_id ), get_the_title( $activity->item_id ) )
		);
	} elseif ( empty( $action ) ) {
		$action = sprintf( esc_html__( '%s added a new listing', 'directorist-buddypress-integration' ), $user_link );
	}

	return $action;
}

function dbp_format_activity_action_share_listing( $action, $activity ) {
	$user_link = bp_core_get_userlink( $activity->user_id );

	if ( $activity->item_id ) {
		$action = sprintf(
			esc_html__( '%s shared a listing - %s', 'directorist-buddypress-integration' ),
			$user_link,
			sprintf( '<a href="%s">%s</a>', get_the_permalink( $activity->item_id ), get_the_title( $activity->item_id ) )
		);
	} elseif ( empty( $action ) ) {
		$action = sprintf( esc_html__( '%s shared a listing', 'directorist-buddypress-integration' ), $user_link );
	}
	
	return $action;
}
