<?php
/**
 * Plugin Name: Directorist - BuddyPress Integration
 * Plugin URI: https://directorist.com/product/directorist-buddypress-integration
 * Description: BuddyPress compatibility and integration extension for <a href="https://directorist.com/" target="blank">Directorist</a>
 * Version: 1.3
 * Author: wpWax
 * Author URI: https://directorist.com/
 * License: GPLv2 or later
 * Text Domain: directorist-buddypress-integration
 * Domain Path: /languages
 */

/**
 * Directorist buddypress extension.
 * 
 * @package wpWax\Directorist\BuddyPress
 */

defined( 'ABSPATH' ) || die();

final class Directorist_BuddyPress_Integration {

	/**
	 * Self instance.
	 *
	 * @var Directorist_BuddyPress_Integration
	 */
	private static $instance = null;

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	public $version = '1.3';

	/**
	 * Plugin directory path.
	 *
	 * @var string
	 */
	public $plugin_dir;

	/**
	 * Plugin directory url.
	 *
	 * @var string
	 */
	public $plugin_url;

	/**
	 * Plugin templates directory path.
	 *
	 * @var string
	 */
	public $template_dir;

	/**
	 * Instantiate or get the the instance.
	 *
	 * @return Directorist_BuddyPress_Integration
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	private function __construct() {
		$this->setup_variables();
		$this->setup_actions();
	}

	protected function setup_variables() {
		$this->plugin_dir   = plugin_dir_path( __FILE__ );
		$this->plugin_url   = plugin_dir_url( __FILE__ );
		$this->template_dir = trailingslashit( $this->plugin_dir . 'includes/templates' );
	}

	protected function setup_actions() {
		add_action( 'bp_include', array( $this, 'on_bp_include' ), 90 );
		add_action( 'plugins_loaded', array( $this, 'on_plugins_loaded' ), 20 );
	}

	public function on_plugins_loaded() {
		// Show directorist missing notice
		if ( ! class_exists( 'Directorist_Base', false ) ) {
			if ( $this->can_see_admin_notice() ) {
				add_action( 'admin_notices', array( $this, 'show_directorist_missing_admin_notice' ) );
			}
			return;
		}

		// Show buddypress missing notice
		if ( ! class_exists( 'BuddyPress', false ) ) {
			if ( $this->can_see_admin_notice() ) {
				add_action( 'admin_notices', array( $this, 'show_buddypress_missing_admin_notice' ) );
			}
			return;
		}

		if ( is_admin() ) {
			include_once $this->plugin_dir . 'includes/class-settings.php';
			include_once $this->plugin_dir . 'includes/class-builder.php';
		}

		add_action( 'template_redirect', array( $this, 'setup_login_registration_redirect' ) );

		$this->load_plugin_textdomain();

		// Add link to settings page.
		add_filter( 'plugin_action_links',               array( $this, 'modify_plugin_action_links' ), 10, 2 );
		add_filter( 'network_admin_plugin_action_links', array( $this, 'modify_plugin_action_links' ), 10, 2 );

		include_once $this->plugin_dir . 'includes/class-template.php';
	}
	
	public function modify_plugin_action_links( $links, $file ) {
		// Return normal links if not BuddyPress.
		if ( plugin_basename( __FILE__ ) != $file ) {
			return $links;
		}

		$url = add_query_arg( array(
			'post_type' => ATBDP_POST_TYPE,
			'page'      => 'atbdp-settings#extension_settings__buddypress-integration'
		), self_admin_url( 'edit.php' ) );

		// Add a few links to the existing links array.
		return array_merge( $links, array(
			'settings' => '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Settings', 'directorist-buddypress-integration' ) . '</a>',
		) );
	}

	protected function load_plugin_textdomain() {
		load_plugin_textdomain(
			'directorist-buddypress-integration',
			false,
			dirname( plugin_basename( __FILE__ ) ) . '/languages'
		 );
	}

	protected function can_see_admin_notice() {
		return ( current_user_can( 'install_plugins' ) || current_user_can( 'activate_plugins' ) );
	}

	public function on_bp_include() {
		if ( ! class_exists( 'Directorist_Base' ) ) {
			if ( $this->can_see_admin_notice() ) {
				add_action( 'admin_notices', array( $this, 'show_directorist_missing_admin_notice' ) );
			}
			return;
		}

		include_once $this->plugin_dir . 'includes/listings-common-functions.php';
		include_once $this->plugin_dir . 'includes/class-bp-listings-component.php';
		include_once $this->plugin_dir . 'includes/class-bp-listings-loader.php';
	}

	/**
	 * Setup registration and login redirection.
	 *
	 * @return void
	 */
	public function setup_login_registration_redirect() {
		$login_page_id = (int) get_directorist_option( 'user_login' );
		$redirect_login = (bool) get_directorist_option( 'redirect_login_to_bb_login', true );

		if ( $redirect_login && $login_page_id && is_page( $login_page_id ) ) {
			wp_safe_redirect( wp_login_url() );
			die();
		}

		$registration_page_id = (int) get_directorist_option( 'custom_registration' );
		$redirect_registration = (bool) get_directorist_option( 'redirect_registration_to_bb_registration', true );
		
		if ( $redirect_registration && $registration_page_id && is_page( $registration_page_id ) ) {
			wp_safe_redirect( wp_registration_url() );
			die();
		}
	}

	/**
	 * Show directorist missing admin notice.
	 * 
	 * When installed but deactivated it'll show an activation link,
	 * and when not installed it'll show an installation link.
	 *
	 * @return void
	 */
	public function show_directorist_missing_admin_notice() {
		$plugin_slug = 'directorist';
		$plugin_file = "{$plugin_slug}/directorist-base.php";

		if ( file_exists( trailingslashit( WP_PLUGIN_DIR ) . $plugin_file ) ) {
			$notice_title = __( 'Activate Directorist', 'directorist-buddypress-integration' );
			$notice_url   = wp_nonce_url(
				"plugins.php?action=activate&plugin={$plugin_file}&plugin_status=all&paged=1",
				"activate-plugin_{$plugin_file}"
			);
		} else {
			$notice_title = __( 'Install Directorist', 'directorist-buddypress-integration' );
			$notice_url = wp_nonce_url(
				self_admin_url( "update.php?action=install-plugin&plugin={$plugin_slug}" ),
				"install-plugin_{$plugin_slug}"
			);
		}

		$notice = wp_kses_data( sprintf(
			/* translators: 1: Plugin name 2: Directorist 3: Directorist installation link */
			__( '%1$s requires %2$s to be installed and activated to function properly. %3$s', 'directorist-buddypress-integration' ),
			'<strong>' . __( 'Directorist BuddyPress Integration', 'directorist-buddypress-integration' ) . '</strong>',
			'<strong>' . __( 'Directorist', 'directorist-buddypress-integration' ) . '</strong>',
			'<a href="' . esc_url( $notice_url ) . '">' . $notice_title . '</a>'
		) );

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $notice );
	}

	/**
	 * Show BuddyPress missing admin notice.
	 * 
	 * When installed but deactivated it'll show an activation link,
	 * and when not installed it'll show an installation link.
	 *
	 * @return void
	 */
	public function show_buddypress_missing_admin_notice() {
		$plugin_slug = 'buddypress';
		$plugin_file = "{$plugin_slug}/bp-loader.php";

		if ( file_exists( trailingslashit( WP_PLUGIN_DIR ) . $plugin_file ) ) {
			$notice_title = __( 'Activate BuddyPress', 'directorist-buddypress-integration' );
			$notice_url   = wp_nonce_url(
				"plugins.php?action=activate&plugin={$plugin_file}&plugin_status=all&paged=1",
				"activate-plugin_{$plugin_file}"
			);
		} else {
			$notice_title = __( 'Install BuddyPress', 'directorist-buddypress-integration' );
			$notice_url = wp_nonce_url(
				self_admin_url( "update.php?action=install-plugin&plugin={$plugin_slug}" ),
				"install-plugin_{$plugin_slug}"
			);
		}

		$notice = wp_kses_data( sprintf(
			/* translators: 1: Plugin name 2: Directorist 3: Directorist installation link */
			__( '%1$s requires %2$s to be installed and activated to function properly. %3$s', 'directorist-buddypress-integration' ),
			'<strong>' . __( 'Directorist BuddyPress Integration', 'directorist-buddypress-integration' ) . '</strong>',
			'<strong>' . __( 'BuddyPress', 'directorist-buddypress-integration' ) . '</strong>',
			'<a href="' . esc_url( $notice_url ) . '">' . $notice_title . '</a>'
		) );

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $notice );
	}
}

function directorist_buddypress() {
	return Directorist_BuddyPress_Integration::get_instance();
}

directorist_buddypress(); // get the plugin running
