<?php

/**
 * Plugin Name: Directorist Listings Slider & Carousel
 * Plugin URI: https://directorist.com/product/directorist-listings-slider-and-carousel
 * Description: This is an extension for Directorist plugin. It helps to create beautiful listings slider and carousel.
 * Version: 1.6.3
 * Author: wpWax
 * Author URI: https://wpwax.com
 * License: GPLv2 or later
 * Text Domain: directorist-slider-carousel
 * Domain Path: /languages
 */

// prevent direct access to the file
defined('ABSPATH') || die('No direct script access allowed!');
if (!class_exists('BD_Slider_Carousel')) {
    final class BD_Slider_Carousel
    {


        /** Singleton *************************************************************/

        /**
         * @var BD_Slider_Carousel The one true BD_Slider_Carousel
         * @since 1.0
         */
        private static $instance;

        /**
         * Main BD_Slider_Carousel Instance.
         *
         * Insures that only one instance of BD_Slider_Carousel exists in memory at any one
         * time. Also prevents needing to define globals all over the place.
         *
         * @since 1.0
         * @static
         * @static_var array $instance
         * @uses BD_Slider_Carousel::setup_constants() Setup the constants needed.
         * @uses BD_Slider_Carousel::load_textdomain() load the language files.
         * @see  BD_Slider_Carousel()
         * @return object|BD_Slider_Carousel The one true BD_Slider_Carousel
         */
        public static function instance()
        {
            if (!isset(self::$instance) && !(self::$instance instanceof BD_Slider_Carousel)) {
                self::$instance = new BD_Slider_Carousel;
                self::$instance->setup_constants();

                add_action('plugins_loaded', array(self::$instance, 'load_textdomain'));
                add_action('admin_enqueue_scripts', array(self::$instance, 'load_needed_scripts'));
                add_shortcode('directorist_slider_carousel', array(self::$instance, 'directorist_slider_carousel'));
                add_filter('atbdp_license_settings_controls', array(self::$instance, 'slider_license_settings_controls'));
                // license and auto update handler
                add_action('wp_ajax_atbdp_slider_license_activation', array(self::$instance, 'atbdp_slider_license_activation'));
                // license deactivation
                add_action('wp_ajax_atbdp_slider_license_deactivation', array(self::$instance, 'atbdp_slider_license_deactivation'));

                add_filter( 'atbdp_listing_type_settings_field_list', array( self::$instance, 'atbdp_listing_type_settings_field_list' ) );
                add_filter( 'atbdp_extension_settings_submenu', array( self::$instance, 'atbdp_extension_settings_submenus' ) );
            }

            return self::$instance;
        }

        private function __construct()
        {
            /*making it private prevents constructing the object*/
        }
        public function atbdp_listing_type_settings_field_list( $booking_fields ) {
            $booking_fields['listing_display_by'] = [
                'label' => __('Display By', BDSC_TEXTDOMAIN),
                        'type'  => 'select',
                        'value' => 'slider',
                        'options' => [
                            [
                                'value' => 'slider',
                                'label' => __('Slider', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'carousel',
                                'label' => __('Carousel', BDSC_TEXTDOMAIN),
                            ],
                        ],
            ];
            $booking_fields['listing_type'] = [
                'label' => __('Listing Type', BDSC_TEXTDOMAIN),
                        'type'  => 'select',
                        'value' => 'latest_listings',
                        'options' => [
                            [
                                'value' => 'a_to_z',
                                'label' => __('A to Z (Title)', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'z_to_a',
                                'label' => __('Z to A (Title)', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'latest_listings',
                                'label' => __('Latest Listings', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'oldest_listings',
                                'label' => __('Oldest Listings', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'popular_listings',
                                'label' => __('Popular Listings', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'price_low_high',
                                'label' => __('Price (low to high)', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'price_high_low',
                                'label' => __('Price (high to low)', BDSC_TEXTDOMAIN),
                            ],
                            [
                                'value' => 'random_listings',
                                'label' => __('Random Listings', BDSC_TEXTDOMAIN),
                            ]
                        ],
            ];
            $booking_fields['number_of_listing'] = [
                'label' => __('Total Number of Listings', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 10,
                'placeholder' => '10',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 50,
                ],
            ];
            $booking_fields['slider_cropping'] = [
                'label'             => __('Slider Image Cropping', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
                'description' => __('If the slider images are not in the same size, it helps automatically resizing.', BDSC_TEXTDOMAIN),
            ];
            $booking_fields['slider_image_width'] = [
                'label' => __('Custom Width', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 668,
                'placeholder' => '668',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 1200,
                ],
            ];
            $booking_fields['slider_image_height'] = [
                'label' => __('Custom Height', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 419,
                'placeholder' => '419',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 1200,
                ],
            ];
            $booking_fields['display_slider_title'] = [
                'label'             => __('Display Title', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['display_slider_excerpt'] = [
                'label'             => __('Excerpt Words Limit', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['excerpt_words_limit'] = [
                'label' => __('Excerpt Words Limit', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 35,
                'placeholder' => '35',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 100,
                ],
            ];
            $booking_fields['display_thumbnail'] = [
                'label'             => __('Display Thumbnail on Slider', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['slider_thumbnail_columns'] = [
                'label' => __('Number of  Thumbnail Columns', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 4,
                'placeholder' => '4',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 10,
                ],
            ];
            $booking_fields['slider_navigation'] = [
                'label'             => __('Display Slider Navigation', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['thumbnail_navigation'] = [
                'label'             => __('Display Thumbnail Navigation', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['slider_autoplay'] = [
                'label'             => __('Autoplay', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['slider_autoplay_speed'] = [
                'label' => __('Autoplay Speed', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 2000,
                'placeholder' => '2000',
                'rules' => [
                    'required' => true,
                    'min' => 1000,
                    'max' => 10000,
                ],
            ];
            $booking_fields['carousel_cropping'] = [
                'label'             => __('Carousel Image Cropping', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'description'       => __('If the slider images are not in the same size, it helps automatically resizing.', BDSC_TEXTDOMAIN),
                'value'             => true,
            ];
            $booking_fields['carousel_image_width'] = [
                'label' => __('Custom Width', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 213,
                'placeholder' => '213',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 1200,
                ],
            ];
            $booking_fields['carousel_image_height'] = [
                'label' => __('Custom Height', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 160,
                'placeholder' => '160',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 1200,
                ],
            ];
            $booking_fields['display_carousel_title'] = [
                'label'             => __('Display Title', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['ls_display_publish_date'] = [
                'label'             => __('Display Publish Date', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['ls_display_category'] = [
                'label'             => __('Display Category', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['carousel_columns_desktop'] = [
                'label' => __('Custom Height', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 4,
                'placeholder' => '4',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 10,
                ],
            ];
            $booking_fields['carousel_columns_tab'] = [
                'label' => __('Number of Columns on Tab', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 2,
                'placeholder' => '2',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 10,
                ],
            ];
            $booking_fields['carousel_columns_mobile'] = [
                'label' => __('Number of Columns on Mobile', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 1,
                'placeholder' => '1',
                'rules' => [
                    'required' => true,
                    'min' => 1,
                    'max' => 10,
                ],
            ];
            $booking_fields['carousel_navigation'] = [
                'label'             => __('Display Carousel Navigation', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['nav_position'] = [
                'label' => __('Navigation Position', BDSC_TEXTDOMAIN),
                'type'  => 'select',
                'value' => 'middle',
                'options' => [
                    [
                        'value' => 'top',
                        'label' => __('Top', BDSC_TEXTDOMAIN),
                    ],
                    [
                        'value' => 'middle',
                        'label' => __('Middle', BDSC_TEXTDOMAIN),
                    ],

                    [
                        'value' => 'default',
                                        'label' => __('Bottom', BDSC_TEXTDOMAIN),
                    ],
                ],
            ];
            $booking_fields['carousel_autoplay'] = [
                'label'             => __('Autoplay', BDSC_TEXTDOMAIN),
                'type'              => 'toggle',
                'value'             => true,
            ];
            $booking_fields['carousel_autoplay_speed'] = [
                'label' => __('Autoplay Speed', BDSC_TEXTDOMAIN),
                'type'  => 'number',
                'value' => 2000,
                'placeholder' => '2000',
                'rules' => [
                    'required' => true,
                    'min' => 1000,
                    'max' => 10000,
                ],
            ];

            return $booking_fields;
        }

        public function atbdp_extension_settings_submenus( $submenu ) {
            $submenu['carousel_submenu'] = [
                'label' => __('Listing Slider & Carousel', 'directorist-booking'),
                        'icon' => '<i class="fa fa-slideshare"></i>',
                        'sections' => apply_filters( 'atbdp_booking_settings_controls', [
                            'general_section' => [
                                'title'       => '',
                                'fields'      =>  [ 'listing_display_by', 'listing_type', 'number_of_listing' ],
                            ],
                            'slider' => [
                                'title'       => __('Listing Slider Settings', 'directorist-booking'),
                                'fields'      =>  [ 'slider_cropping', 'slider_image_width', 'slider_image_height', 'display_slider_title', 'excerpt_words_limit', 'display_thumbnail', 'slider_thumbnail_columns', 'slider_navigation', 'thumbnail_navigation', 'slider_autoplay', 'slider_autoplay_speed' ],
                            ],
                            'carousel' => [
                                'title'       => __('Listings Carousel Settings', 'directorist-booking'),
                                'fields'      =>  [ 'carousel_cropping', 'carousel_image_width', 'carousel_image_height', 'display_carousel_title', 'ls_display_publish_date', 'ls_display_category', 'carousel_columns_desktop', 'carousel_columns_tab', 'carousel_columns_mobile', 'carousel_navigation', 'nav_position', 'carousel_autoplay', 'carousel_autoplay_speed' ],
                            ],
                        ] ),
            ];

            return $submenu;
        }

        public function atbdp_slider_license_deactivation()
        {
            $license = !empty($_POST['slider_license']) ? trim($_POST['slider_license']) : '';
            $options = get_option('atbdp_option');
            $options['slider_license'] = $license;
            update_option('atbdp_option', $options);
            update_option('directorist_slider_license', $license);
            $data = array();
            if (!empty($license)) {
                // data to send in our API request
                $api_params = array(
                    'edd_action' => 'deactivate_license',
                    'license'    => $license,
                    'item_id'    => ATBDP_SLIDER_POST_ID, // The ID of the item in EDD
                    'url'        => home_url()
                );
                // Call the custom API.
                $response = wp_remote_post(ATBDP_AUTHOR_URL, array('timeout' => 15, 'sslverify' => false, 'body' => $api_params));
                // make sure the response came back okay
                if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code($response)) {

                    $data['msg'] =  (is_wp_error($response) && !empty($response->get_error_message())) ? $response->get_error_message() : __('An error occurred, please try again.', 'directorist-slider-carousel');
                    $data['status'] = false;
                } else {

                    $license_data = json_decode(wp_remote_retrieve_body($response));
                    if (!$license_data) {
                        $data['status'] = false;
                        $data['msg'] = __('Response not found!', 'directorist-slider-carousel');
                        wp_send_json($data);
                        die();
                    }
                    update_option('directorist_slider_license_status', $license_data->license);
                    if (false === $license_data->success) {
                        switch ($license_data->error) {
                            case 'expired':
                                $data['msg'] = sprintf(
                                    __('Your license key expired on %s.', 'directorist-slider-carousel'),
                                    date_i18n(get_option('date_format'), strtotime($license_data->expires, current_time('timestamp')))
                                );
                                $data['status'] = false;
                                break;

                            case 'revoked':
                                $data['status'] = false;
                                $data['msg'] = __('Your license key has been disabled.', 'directorist-slider-carousel');
                                break;

                            case 'missing':

                                $data['msg'] = __('Invalid license.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;

                            case 'invalid':
                            case 'site_inactive':

                                $data['msg'] = __('Your license is not active for this URL.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;

                            case 'item_name_mismatch':

                                $data['msg'] = sprintf(__('This appears to be an invalid license key for %s.', 'directorist-slider-carousel'), 'Directorist - Slider & Carousel');
                                $data['status'] = false;
                                break;

                            case 'no_activations_left':

                                $data['msg'] = __('Your license key has reached its activation limit.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;

                            default:
                                $data['msg'] = __('An error occurred, please try again.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;
                        }
                    } else {
                        $data['status'] = true;
                        $data['msg'] = __('License deactivated successfully!', 'directorist-slider-carousel');
                    }
                }
            } else {
                $data['status'] = false;
                $data['msg'] = __('License not found!', 'directorist-slider-carousel');
            }
            wp_send_json($data);
            die();
        }

        public function atbdp_slider_license_activation()
        {
            $license = !empty($_POST['slider_license']) ? trim($_POST['slider_license']) : '';
            $options = get_option('atbdp_option');
            $options['slider_license'] = $license;
            update_option('atbdp_option', $options);
            update_option('directorist_slider_license', $license);
            $data = array();
            if (!empty($license)) {
                // data to send in our API request
                $api_params = array(
                    'edd_action' => 'activate_license',
                    'license'    => $license,
                    'item_id'    => ATBDP_SLIDER_POST_ID, // The ID of the item in EDD
                    'url'        => home_url()
                );
                // Call the custom API.
                $response = wp_remote_post(ATBDP_AUTHOR_URL, array('timeout' => 15, 'sslverify' => false, 'body' => $api_params));
                // make sure the response came back okay
                if (is_wp_error($response) || 200 !== wp_remote_retrieve_response_code($response)) {

                    $data['msg'] =  (is_wp_error($response) && !empty($response->get_error_message())) ? $response->get_error_message() : __('An error occurred, please try again.', 'directorist-slider-carousel');
                    $data['status'] = false;
                } else {

                    $license_data = json_decode(wp_remote_retrieve_body($response));
                    if (!$license_data) {
                        $data['status'] = false;
                        $data['msg'] = __('Response not found!', 'directorist-slider-carousel');
                        wp_send_json($data);
                        die();
                    }
                    update_option('directorist_slider_license_status', $license_data->license);
                    if (false === $license_data->success) {
                        switch ($license_data->error) {
                            case 'expired':
                                $data['msg'] = sprintf(
                                    __('Your license key expired on %s.', 'directorist-slider-carousel'),
                                    date_i18n(get_option('date_format'), strtotime($license_data->expires, current_time('timestamp')))
                                );
                                $data['status'] = false;
                                break;

                            case 'revoked':
                                $data['status'] = false;
                                $data['msg'] = __('Your license key has been disabled.', 'directorist-slider-carousel');
                                break;

                            case 'missing':

                                $data['msg'] = __('Invalid license.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;

                            case 'invalid':
                            case 'site_inactive':

                                $data['msg'] = __('Your license is not active for this URL.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;

                            case 'item_name_mismatch':

                                $data['msg'] = sprintf(__('This appears to be an invalid license key for %s.', 'directorist-slider-carousel'), 'Directorist - Slider & Carousel');
                                $data['status'] = false;
                                break;

                            case 'no_activations_left':

                                $data['msg'] = __('Your license key has reached its activation limit.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;

                            default:
                                $data['msg'] = __('An error occurred, please try again.', 'directorist-slider-carousel');
                                $data['status'] = false;
                                break;
                        }
                    } else {
                        $data['status'] = true;
                        $data['msg'] = __('License activated successfully!', 'directorist-slider-carousel');
                    }
                }
            } else {
                $data['status'] = false;
                $data['msg'] = __('License not found!', 'directorist-slider-carousel');
            }
            wp_send_json($data);
            die();
        }

        public function load_needed_scripts($screen)
        {
            if (isset($_GET['page']) && ('aazztech_settings' === $_GET['page'])) {
                wp_enqueue_style('admin_slider_main_css', plugin_dir_url(__FILE__) . 'assets/admin/style.css');
                wp_enqueue_script('slider_main_js', plugin_dir_url(__FILE__) . 'assets/admin/main.js', array('jquery'));
                wp_localize_script('slider_main_js', 'slider_js_obj', array('ajaxurl' => admin_url('admin-ajax.php')));
            }
        }

        public function slider_license_settings_controls($default)
        {
            $status = get_option('directorist_slider_license_status');
            if (!empty($status) && ($status !== false && $status == 'valid')) {
                $action = array(
                    'type' => 'toggle',
                    'name' => 'slider_deactivated',
                    'label' => __('Action', 'directorist-slider-carousel'),
                    'validation' => 'numeric',
                );
            } else {
                $action = array(
                    'type' => 'toggle',
                    'name' => 'slider_activated',
                    'label' => __('Action', 'directorist-slider-carousel'),
                    'validation' => 'numeric',
                );
            }
            $new = apply_filters('atbdp_slider_license_controls', array(
                'type' => 'section',
                'title' => __('Slider & Carousel', 'directorist-slider-carousel'),
                'description' => __('You can active your Slider & Carousel extension here.', 'directorist-slider-carousel'),
                'fields' => apply_filters('atbdp_slider_license_settings_field', array(
                    array(
                        'type' => 'textbox',
                        'name' => 'slider_license',
                        'label' => __('License', 'directorist-slider-carousel'),
                        'description' => __('Enter your Slider & Carousel extension license', 'directorist-slider-carousel'),
                        'default' => '',
                    ),
                    $action,
                )),
            ));
            $settings = apply_filters('atbdp_licence_menu_for_listing_slider', true);
            if ($settings) {
                array_push($default, $new);
            }
            return $default;
        }

        public function directorist_slider_carousel($atts)
        {

            if (class_exists('Directorist_Base')) {
                $atbdp_legacy_template = get_directorist_option( 'atbdp_legacy_template', false );
                wp_enqueue_script('atbdp-bootstrap-script');
                if( $atbdp_legacy_template ) {
                    wp_enqueue_style('slickcss');
                } else {
                    wp_enqueue_script( 'directorist-slick' );
                    wp_enqueue_style('directorist-slick-style');
                }

            }
            wp_enqueue_style('stylecss', plugin_dir_url(__FILE__) . '/assets/css/style.css');
            //wp_enqueue_style('bootstrap-mincss',plugin_dir_url(__FILE__) . '/assets/css/bootstrap.min.css');
            //wp_enqueue_style('slickcss',plugin_dir_url(__FILE__) . '/assets/css/slick.css');
            //wp_enqueue_script('bootstrapjs',plugin_dir_url(__FILE__) . '/assets/js/bootstrap.min.js',array('jquery'));
            //wp_enqueue_script('propper',plugin_dir_url(__FILE__) . '/assets/js/proper.js',array('jquery'));

            $display_by                              = get_directorist_option('listing_display_by', 'slider');
            $listing_type                            = get_directorist_option('listing_type', 'latest_listings');
            $slider_cropping                         = get_directorist_option('slider_cropping', 1);
            $slider_image_width                      = get_directorist_option('slider_image_width', 668);
            $slider_image_height                     = get_directorist_option('slider_image_height', 419);
            $display_slider_title                    = get_directorist_option('display_slider_title', 1);
            $display_slider_excerpt                  = get_directorist_option('display_slider_excerpt', 1);
            $excerpt_words_limit                     = get_directorist_option('excerpt_words_limit', 35);
            $display_thumbnail                       = get_directorist_option('display_thumbnail', 1);
            $slider_thumbnail_columns                = get_directorist_option('slider_thumbnail_columns', 4);
            $slider_navigation                       = get_directorist_option('slider_navigation', 1);
            $thumbnail_navigation                    = get_directorist_option('thumbnail_navigation', 1);
            $carousel_cropping                       = get_directorist_option('carousel_cropping', 1);
            $carousel_image_width                    = get_directorist_option('carousel_image_width', 213);
            $carousel_image_height                   = get_directorist_option('carousel_image_height', 160);
            $display_carousel_title                  = get_directorist_option('display_carousel_title', 1);
            $display_carousel_publish_date           = get_directorist_option('ls_display_publish_date', 1);
            $display_carousel_category               = get_directorist_option('ls_display_category', 1);
            $carousel_columns_desktop                = get_directorist_option('carousel_columns_desktop', 4);
            $carousel_columns_tab                    = get_directorist_option('carousel_columns_tab', 2);
            $carousel_columns_mobile                 = get_directorist_option('carousel_columns_mobile', 1);
            $carousel_navigation                     = get_directorist_option('carousel_navigation', 1);
            $carousel_navigation_position            = get_directorist_option('nav_position', 'middle');
            $has_featured                            = get_directorist_option('enable_featured_listing');
            $number_of_listing                       = get_directorist_option('number_of_listing', 10);
            $slider_autoplay                         = get_directorist_option('slider_autoplay', 1);
            $slider_autoplay_speed                   = get_directorist_option('slider_autoplay_speed', 2000);
            $carousel_autoplay                       = get_directorist_option('carousel_autoplay', 1);
            $carousel_autoplay_speed                 = get_directorist_option('carousel_autoplay_speed', 2000);
            ob_start();
            $atts = shortcode_atts(array(
                'directory_type'                        => '',
                'display_by'                            => !empty($display_by) ? $display_by : 'slider',
                'listing_type'                          => !empty($listing_type) ? $listing_type : 'latest_listings',
                'slider_cropping'                       => !empty($slider_cropping) ? 'yes' : 'no',
                'slider_image_width'                    => !empty($slider_image_width) ? $slider_image_width : 668,
                'slider_image_height'                   => !empty($slider_image_height) ? $slider_image_height : 419,
                'display_slider_title'                  => !empty($display_slider_title) ? 'yes' : 'no',
                'display_slider_excerpt'                => !empty($display_slider_excerpt) ? 'yes' : 'no',
                'excerpt_words_limit'                   => !empty($excerpt_words_limit) ? $excerpt_words_limit : 35,
                'display_thumbnail'                     => !empty($display_thumbnail) ? 'yes' : 'no',
                'slider_thumbnail_columns'              => !empty($slider_thumbnail_columns) ? $slider_thumbnail_columns : 4,
                'slider_navigation'                     => !empty($slider_navigation) ? 'yes' : 'no',
                'thumbnail_navigation'                  => !empty($thumbnail_navigation) ? 'yes' : 'no',
                'carousel_cropping'                     => !empty($carousel_cropping) ? 'yes' : 'no',
                'carousel_image_width'                  => !empty($carousel_image_width) ? $carousel_image_width : 213,
                'carousel_image_height'                 => !empty($carousel_image_height) ? $carousel_image_height : 160,
                'display_carousel_title'                => !empty($display_carousel_title) ? 'yes' : 'no',
                'display_carousel_publish_date'         => !empty($display_carousel_publish_date) ? 'yes' : 'no',
                'display_carousel_category'             => !empty($display_carousel_category) ? 'yes' : 'no',
                'carousel_columns_desktop'              => !empty($carousel_columns_desktop) ? $carousel_columns_desktop : 4,
                'carousel_columns_tab'                  => !empty($carousel_columns_tab) ? $carousel_columns_tab : 2,
                'carousel_columns_mobile'               => !empty($carousel_columns_mobile) ? $carousel_columns_mobile : 1,
                'carousel_navigation'                   => !empty($carousel_navigation) ? 'yes' : 'no',
                'carousel_navigation_position'          => !empty($carousel_navigation_position) ? $carousel_navigation_position : 'middle',
                'categories'                            => '',
                'locations'                             => '',
                'tags'                                  => '',
                'ids'                                   => '',
                'featured'                              => !empty($has_featured) ? 'yes' : 'no',
                'number_of_listing'                     => !empty($number_of_listing) ? $number_of_listing : 10,
                'featured_only'                         => 'no'
            ), $atts);
            $parameter = $atts;
            extract( $parameter );
            $directory_type = !empty( $directory_type ) ? explode( ',', $directory_type ) : '';
            $categories     = !empty($categories) ? explode(',', $categories) : '';
            $locations      = !empty($locations) ? explode(',', $locations) : '';
            $tags           = !empty($tags) ? explode(',', $tags) : '';
            $ids            = !empty($ids) ? explode(',', $ids) : '';
            $args = array(
                'post_type'      => 'at_biz_dir',
                'post_status'    => 'publish',
                'posts_per_page' => $number_of_listing,
            );

            $listingbyid_arg = array();

            if (!empty($ids)) {
                $listingbyid_arg = $ids;
                $args['orderby'] = 'post__in';
                $args['post__in'] = $listingbyid_arg;
            } else {
                $tax_queries = array(); // initiate the tax query var to append to it different tax query

                if ( ! empty( $categories ) && ! empty( $locations ) && ! empty( $tags ) && ! empty( $directory_type ) ) {

                    $tax_queries['tax_query'] = array(
                        'relation' => 'AND',
                        array(
                            'taxonomy'         => ATBDP_DIRECTORY_TYPE,
                            'field'            => 'slug',
                            'terms'            => ! empty( $directory_type ) ? $directory_type : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_CATEGORY,
                            'field'            => 'slug',
                            'terms'            => !empty($categories) ? $categories : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_LOCATION,
                            'field'            => 'slug',
                            'terms'            => !empty($locations) ? $locations : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_TAGS,
                            'field'            => 'slug',
                            'terms'            => !empty($tags) ? $tags : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),

                    );
                } elseif ( ! empty( $categories ) && ! empty( $directory_type ) ) {

                    $tax_queries['tax_query'] = array(
                        'relation' => 'AND',
                        array(
                            'taxonomy'         => ATBDP_CATEGORY,
                            'field'            => 'slug',
                            'terms'            => !empty($categories) ? $categories : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_DIRECTORY_TYPE,
                            'field'            => 'slug',
                            'terms'            => ! empty( $directory_type ) ? $directory_type : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),

                    );
                } elseif (!empty($directory_type) && !empty($tags)) {
                    $tax_queries['tax_query'] = array(
                        'relation' => 'AND',
                        array(
                            'taxonomy'         => ATBDP_DIRECTORY_TYPE,
                            'field'            => 'slug',
                            'terms'            => ! empty( $directory_type ) ? $directory_type : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_TAGS,
                            'field'            => 'slug',
                            'terms'            => !empty($tags) ? $tags : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),

                    );
                }
                elseif (!empty($categories) && !empty($tags)) {
                    $tax_queries['tax_query'] = array(
                        'relation' => 'AND',
                        array(
                            'taxonomy'         => ATBDP_CATEGORY,
                            'field'            => 'slug',
                            'terms'            => !empty($categories) ? $categories : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_TAGS,
                            'field'            => 'slug',
                            'terms'            => !empty($tags) ? $tags : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),

                    );
                } elseif (!empty($directory_type) && !empty($locations)) {
                    $tax_queries['tax_query'] = array(
                        'relation' => 'AND',
                        array(
                            'taxonomy'         => ATBDP_DIRECTORY_TYPE,
                            'field'            => 'slug',
                            'terms'            => ! empty( $directory_type ) ? $directory_type : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_LOCATION,
                            'field'            => 'slug',
                            'terms'            => !empty($locations) ? $locations : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),

                    );
                } elseif (!empty($categories) && !empty($locations)) {
                    $tax_queries['tax_query'] = array(
                        'relation' => 'AND',
                        array(
                            'taxonomy'         => ATBDP_CATEGORY,
                            'field'            => 'slug',
                            'terms'            => !empty($categories) ? $categories : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_LOCATION,
                            'field'            => 'slug',
                            'terms'            => !empty($locations) ? $locations : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),

                    );
                } elseif (!empty($tags) && !empty($locations)) {
                    $tax_queries['tax_query'] = array(
                        'relation' => 'AND',
                        array(
                            'taxonomy'         => ATBDP_TAGS,
                            'field'            => 'slug',
                            'terms'            => !empty($tags) ? $tags : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),
                        array(
                            'taxonomy'         => ATBDP_LOCATION,
                            'field'            => 'slug',
                            'terms'            => !empty($locations) ? $locations : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        ),

                    );
                } elseif (! empty( $directory_type ) ) {
                    $tax_queries['tax_query'] = array(
                        array(
                            'taxonomy'         => ATBDP_DIRECTORY_TYPE,
                            'field'            => 'slug',
                            'terms'            => ! empty( $directory_type ) ? $directory_type : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        )
                        );
                } elseif (!empty($categories)) {
                    $tax_queries['tax_query'] = array(
                        array(
                            'taxonomy'         => ATBDP_CATEGORY,
                            'field'            => 'slug',
                            'terms'            => !empty($categories) ? $categories : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        )
                    );
                } elseif (!empty($tags)) {
                    $tax_queries['tax_query'] = array(
                        array(
                            'taxonomy'         => ATBDP_TAGS,
                            'field'            => 'slug',
                            'terms'            => !empty($tags) ? $tags : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        )
                    );
                } elseif (!empty($locations)) {
                    $tax_queries['tax_query'] = array(
                        array(
                            'taxonomy'         => ATBDP_LOCATION,
                            'field'            => 'slug',
                            'terms'            => !empty($locations) ? $locations : array(),
                            'include_children' => true, /*@todo; Add option to include children or exclude it*/
                        )
                    );
                }
                $args['tax_query'] = $tax_queries;
                if ((!empty($featured_only)) && ('yes' == $featured_only)) {
                    $meta_queries['_featured'] = array(
                        'key' => '_featured',
                        'value' => 1,
                        'compare' => '='
                    );
                }
                switch ($listing_type) {
                    case 'a_to_z':
                        if ('yes' == $featured) {
                            $args['meta_key'] = '_featured';
                            $args['orderby']  = array(
                                'meta_value_num' => 'DESC',
                                'title'          => 'ASC',
                            );
                        } else {
                            $args['orderby'] = 'title';
                            $args['order']   = 'ASC';
                        };
                        break;
                    case 'z_to_a':
                        if ('yes' == $featured) {
                            $args['meta_key'] = '_featured';
                            $args['orderby']  = array(
                                'meta_value_num' => 'DESC',
                                'title'          => 'DESC',
                            );
                        } else {
                            $args['orderby'] = 'title';
                            $args['order']   = 'DESC';
                        };
                        break;
                    case 'latest_listings':
                        if ('yes' == $featured) {
                            $args['meta_key'] = '_featured';
                            $args['orderby']  = array(
                                'meta_value_num' => 'DESC',
                                'date'           => 'DESC',
                            );
                        } else {
                            $args['orderby'] = 'date';
                            $args['order']   = 'DESC';
                        };
                        break;
                    case 'oldest_listings':
                        if ('yes' == $featured) {
                            $args['meta_key'] = '_featured';
                            $args['orderby']  = array(
                                'meta_value_num' => 'DESC',
                                'date'           => 'ASC',
                            );
                        } else {
                            $args['orderby'] = 'date';
                            $args['order']   = 'ASC';
                        };
                        break;
                    case 'price_low_high':
                        if ('yes' == $featured) {
                            $meta_queries['price'] = array(
                                'key'     => '_price',
                                'type'    => 'NUMERIC',
                                'compare' => 'EXISTS',
                            );

                            $args['orderby']  = array(
                                '_featured' => 'DESC',
                                'price'    => 'ASC',
                            );
                        } else {
                            $args['meta_key'] = '_price';
                            $args['orderby']  = 'meta_value_num';
                            $args['order']    = 'ASC';
                        };
                        break;
                    case 'price_high_low':
                        if ('yes' == $featured) {
                            $meta_queries['price'] = array(
                                'key'     => '_price',
                                'type'    => 'NUMERIC',
                                'compare' => 'EXISTS',
                            );

                            $args['orderby']  = array(
                                '_featured' => 'DESC',
                                'price'     => 'DESC',
                            );
                        } else {
                            $args['meta_key'] = '_price';
                            $args['orderby']  = 'meta_value_num';
                            $args['order']    = 'DESC';
                        };
                        break;
                    case 'popular_listings':
                        if ('yes' == $featured) {
                            $meta_queries['views'] = array(
                                'key'     => '_atbdp_post_views_count',
                                'type'    => 'NUMERIC',
                                'compare' => 'EXISTS',
                            );

                            $args['orderby']  = array(
                                '_featured'                  => 'DESC',
                                '_atbdp_post_views_count'    => 'DESC',
                            );
                        } else {
                            $args['meta_key'] = '_atbdp_post_views_count';
                            $args['orderby']  = 'meta_value_num';
                            $args['order']    = 'DESC';
                        };
                        break;
                    case 'random_listings':
                        if ('yes' == $featured) {
                            $args['meta_key'] = '_featured';
                            $args['orderby']  = 'meta_value_num rand';
                        } else {
                            $args['orderby']  = 'rand';
                        };
                        break;
                }
            }
            if (!empty($meta_queries)) {
                $count_meta_queries = count($meta_queries);
            }

            if (!empty($count_meta_queries)) {
                $args['meta_query'] = ($count_meta_queries > 1) ? array_merge(array('relation' => 'AND'), $meta_queries) : $meta_queries;
            }

            $all_listings = new WP_Query($args);

            $listing_img                    = get_post_meta(get_the_ID(), '_listing_img', true);
            $listing_prv_img                = get_post_meta(get_the_ID(), '_listing_prv_img', true);
            $excerpt                        = get_post_meta(get_the_ID(), '_excerpt', true);
            $default_image_attachment_id                  = get_directorist_option('preview_image', ATBDP_PUBLIC_ASSETS . 'images/grid.jpg');

            //$default_image_attachment_id   = attachment_url_to_postid( $default_image['url'] );
            if ('slider' == $display_by) {
?>
                <div class="directorist-gallery-slider-one">
                    <div class="directorist-slider-for">
                        <?php if ($all_listings->have_posts()) {
                            while ($all_listings->have_posts()) {
                                $all_listings->the_post();
                                $listing_img                    = get_post_meta(get_the_ID(), '_listing_img', true);
                                $listing_prv_img                = get_post_meta(get_the_ID(), '_listing_prv_img', true);
                                $excerpt                        = get_post_meta(get_the_ID(), '_excerpt', true);

                                if (!empty($listing_prv_img)) {

                                    if ('yes' == $slider_cropping) {

                                        $prv_image = atbdp_image_cropping($listing_prv_img, $slider_image_width, $slider_image_height, true, 100)['url'];
                                    } else {
                                        $prv_image   = wp_get_attachment_image_src($listing_prv_img, 'large')[0];
                                    }
                                }
                                if (!empty($listing_img[0])) {
                                    if ('yes' == $slider_cropping) {
                                        $gallery_img = atbdp_image_cropping($listing_img[0], $slider_image_width, $slider_image_height, true, 100)['url'];
                                    } else {
                                        $gallery_img = wp_get_attachment_image_src($listing_img[0], 'large')[0];
                                    }
                                }
                                if ('yes' == $slider_cropping) {

                                    $default = atbdp_image_cropping($default_image_attachment_id, $slider_image_width, $slider_image_height, true, 100)['url'];
                                } else {
                                    $default   = $default_image_attachment_id;
                                }
                        ?>
                                <div>
                                    <figure>
                                        <?php if (!empty($listing_prv_img)) {

                                            echo '<a href="' . esc_url( get_the_permalink() ) . '"><img src="' . esc_url( $prv_image ) . '" alt="listing image" class="directorist-img-fluid"></a>';
                                        } elseif (!empty($listing_img[0]) && empty($listing_prv_img)) {

                                            echo '<a href="' . esc_url( get_the_permalink() ) . '"><img src="' . esc_url($gallery_img) . '" alt="listing image" class="directorist-img-fluid"></a>';
                                        } else {
                                            echo '<a href="' . esc_url( get_the_permalink() ) . '"><img src="' . esc_url( $default_image_attachment_id ) . '" alt="listing image" class="directorist-img-fluid"></a>';
                                        }
                                        ?>
                                        <figcaption>
                                            <?php if ('yes' == $display_slider_title) { ?>
                                                <h2><a href="<?php echo esc_url(get_post_permalink(get_the_ID())); ?>"><?php echo esc_html(stripslashes(get_the_title())); ?></a></h2>
                                            <?php } ?>
                                            <?php if ('yes' == $display_slider_excerpt && !empty($excerpt)) { ?>
                                                <p><?php echo esc_html(stripslashes(wp_trim_words($excerpt, $excerpt_words_limit))); ?></p>
                                            <?php } ?>
                                        </figcaption>
                                    </figure>
                                </div>
                        <?php }
                            wp_reset_postdata();
                        } ?>
                    </div><!-- ends: .directorist-slider-for -->

                    <div class="directorist-slider-nav">
                        <?php if ($all_listings->have_posts()) {
                            while ($all_listings->have_posts()) {
                                $all_listings->the_post();
                                $listing_img                    = get_post_meta(get_the_ID(), '_listing_img', true);
                                $listing_prv_img                = get_post_meta(get_the_ID(), '_listing_prv_img', true);
                                if (!empty($listing_prv_img)) {

                                    $prv_image = atbdp_image_cropping($listing_prv_img, '135', '106', true, 100)['url'];
                                }
                                if (!empty($listing_img[0])) {

                                    $gallery_img = atbdp_image_cropping($listing_img[0], '135', '106', true, 100)['url'];
                                }
                                if ('yes' == $slider_cropping) {

                                    $default = atbdp_image_cropping($default_image_attachment_id, '135', '106', true, 100)['url'];
                                } else {
                                    $default   = $default_image_attachment_id;
                                }
                        ?>
                                <?php if ('yes' == $display_thumbnail) { ?>
                                    <div class="slick-nav-item">
                                        <?php
                                        if (!empty($listing_prv_img)) {

                                            echo '<img src="' . esc_url($prv_image) . '" alt="listing image" class="directorist-img-fluid">';
                                        } elseif (!empty($listing_img[0]) && empty($listing_prv_img)) {

                                            echo '<img src="' . esc_url($gallery_img) . '" alt="listing image" class="directorist-img-fluid">';
                                        } else {
                                            echo '<img src="' . esc_url( $default_image_attachment_id ) . '" alt="listing image" class="directorist-img-fluid">';
                                        }

                                        ?>
                                    </div>
                        <?php }
                            }
                            wp_reset_postdata();
                        } ?>
                    </div><!-- ends: .directorist-slider-nav -->

                    <div class="current-slide"></div>
                </div>
                <script>
                    (function($) {
                        jQuery(document).ready(function() {
                            "use strict";
                            //gallery one slider
                            $('.directorist-slider-for').slick({
                                slidesToShow: 1,
                                slidesToScroll: 1,
                                autoplay: <?php echo esc_js( ! empty( $slider_autoplay ) ? 'true' : 'false' ); ?>,
                                autoplaySpeed: <?php echo esc_js( ! empty( $slider_autoplay_speed ) ? $slider_autoplay_speed : 2000 ); ?>,
                                dots: false,
                                arrows: <?php echo esc_js( ('yes' == $slider_navigation) ? 'true' : 'false' ); ?>,
                                prevArrow: '<span class="icon icon-30x30 directorist-rounded-circle directorist-light-border slick-arrow-left"><?php directorist_icon( 'fas fa-angle-left' ); ?></span>',
                                nextArrow: '<span class="icon icon-30x30 directorist-rounded-circle directorist-light-border slick-arrow-right"><?php directorist_icon( 'fas fa-angle-right' ); ?></span>',
                                fade: true,
                                asNavFor: '.directorist-slider-nav'
                            });
                            $('.directorist-slider-nav').slick({
                                slidesToShow: <?php echo esc_js( $slider_thumbnail_columns ); ?>,
                                slidesToScroll: 1,
                                asNavFor: '.directorist-slider-for',
                                dots: false,
                                arrows: <?php echo ('yes' == $thumbnail_navigation) ? 'true' : 'false'; ?>,
                                prevArrow: '<span class="icon icon-30x30 directorist-rounded-circle directorist-light-border slick-arrow-left"><?php directorist_icon( 'fas fa-angle-left' ); ?></span>',
                                nextArrow: '<span class="icon icon-30x30 directorist-rounded-circle directorist-light-border slick-arrow-right"><?php directorist_icon( 'fas fa-angle-right' ); ?></span>',
                                centerMode: true,
                                focusOnSelect: true,
                                centerPadding: "0",

                            });

                        });
                    })(jQuery);
                </script>
            <?php
            } elseif ('carousel' == $display_by) { ?>


                <div class="directorist-gallery-carousel-wrap directorist-gc-nav-<?php echo esc_attr( $carousel_navigation_position ); ?>">
                    <!-- directorist-gc-nav-default / directorist-gc-nav-top / directorist-gc-nav-mid -->
                    <div class="directorist-gallery-carousel">
                        <?php
                        if ($all_listings->have_posts()) {
                            while ($all_listings->have_posts()) {
                                $all_listings->the_post();
                                $listing_img                    = get_post_meta(get_the_ID(), '_listing_img', true);
                                $listing_prv_img                = get_post_meta(get_the_ID(), '_listing_prv_img', true);
                                $cats                           =  get_the_terms(get_the_ID(), ATBDP_CATEGORY);
                                if (!empty($listing_prv_img)) {

                                    if ('yes' == $carousel_cropping) {

                                        $prv_image = atbdp_image_cropping($listing_prv_img, $carousel_image_width, $carousel_image_height, true, 100)['url'];
                                    } else {
                                        $prv_image   = wp_get_attachment_image_src($listing_prv_img, 'large')[0];
                                    }
                                }
                                if (!empty($listing_img[0])) {
                                    if ('yes' == $carousel_cropping) {

                                        $gallery_img = atbdp_image_cropping($listing_img[0], $carousel_image_width, $carousel_image_height, true, 100)['url'];
                                    } else {
                                        $gallery_img = wp_get_attachment_image_src($listing_img[0], 'large')[0];
                                    }
                                }
                                if ('yes' == $slider_cropping) {

                                    $default = atbdp_image_cropping($default_image_attachment_id, $carousel_image_width, $carousel_image_height, true, 100)['url'];
                                } else {
                                    $default   = wp_get_attachment_image_src($default_image_attachment_id, 'large')[0];
                                }
                        ?>
                                <div class="directorist-gallery-carousel__item">
                                    <figure>
                                        <?php
                                        if (!empty($listing_prv_img)) {

                                            echo '<a href="' . esc_url( get_the_permalink() ) . '"><img src="' . esc_url($prv_image) . '" alt="listing image" class="directorist-img-fluid"></a>';
                                        } elseif (!empty($listing_img[0]) && empty($listing_prv_img)) {

                                            echo '<a href="' . esc_url( get_the_permalink() ) . '"><img src="' . esc_url($gallery_img) . '" alt="listing image" class="directorist-img-fluid"></a>';
                                        } else {
                                            echo '<a href="' . esc_url( get_the_permalink() ) . '"><img src="' . esc_url( $default_image_attachment_id ) . '" alt="listing image" class="directorist-img-fluid"></a>';
                                        }

                                        ?>
                                        <figcaption>
                                            <?php if ('yes' == $display_carousel_title) { ?>
                                                <h3><a href="<?= esc_url(get_post_permalink(get_the_ID())); ?>"><?php echo esc_html(stripslashes(get_the_title())); ?></a></h3>
                                            <?php } ?>
                                            <?php if ('yes' == $display_carousel_publish_date) { ?>
                                                <span class="directorist-post-date"><?php printf(esc_html__('Posted %s ago', BDSC_TEXTDOMAIN), esc_html( human_time_diff(get_the_time('U') ), current_time('timestamp'))); ?></span>
                                            <?php } ?>
                                        </figcaption>
                                    </figure>
                                    <?php if (!empty($cats) && 'yes' == $display_carousel_category) { ?>
                                        <a href="<?php echo esc_url(ATBDP_Permalink::get_category_archive($cats[0])); ?>" class="directorist-cat"><?php echo esc_html( $cats[0]->name ); ?></a>
                                    <?php } ?>
                                </div><!-- ends: .directorist-gallery-carousel__item -->
                        <?php }
                        wp_reset_postdata();
                        } ?>


                    </div>
                    <div class="directorist-gallery-carousel-nav"></div>
                </div>


                <script>
                    (function($) {
                        jQuery(document).ready(function() {
                            $('.directorist-gallery-carousel').each(function (id, elm) {
                                $(elm).slick({
                                    dots: false,
                                    infinite: false,
                                    autoplay: <?php echo esc_js( ! empty( $carousel_autoplay ) ? 'true' : 'false' ); ?>,
                                    autoplaySpeed: <?php echo esc_js( ! empty( $carousel_autoplay_speed ) ? $carousel_autoplay_speed : 2000 ); ?>,
                                    arrows: <?php echo esc_js( ( 'yes' == $carousel_navigation ) ? 'true' : 'false' ); ?>,
                                    prevArrow: '<span class="slick-arrow-left"><?php directorist_icon( 'fas fa-angle-left' ); ?></span>',
                                    nextArrow: '<span class="slick-arrow-right"><?php directorist_icon( 'fas fa-angle-right' ); ?></span>',
                                    appendArrows: $(elm).siblings(".directorist-gallery-carousel-nav"),
                                    speed: 300,
                                    slidesToShow: <?php echo esc_js( ! empty( $carousel_columns_desktop ) ? $carousel_columns_desktop : 4 ); ?>,
                                    slidesToScroll: 1,
                                    responsive: [{
                                            breakpoint: 1024,
                                            settings: {
                                                slidesToShow: <?php echo esc_js( ! empty( $carousel_columns_desktop ) ? $carousel_columns_desktop : 4 ); ?>,
                                                slidesToScroll: 3,
                                                infinite: true,
                                                dots: false
                                            }
                                        },
                                        {
                                            breakpoint: 600,
                                            settings: {
                                                slidesToShow: <?php echo esc_js( ! empty( $carousel_columns_tab ) ? $carousel_columns_tab : 2 ); ?>,
                                                slidesToScroll: 2
                                            }
                                        },
                                        {
                                            breakpoint: 480,
                                            settings: {
                                                slidesToShow: <?php echo esc_js( ! empty( $carousel_columns_mobile ) ? $carousel_columns_mobile : 1 ); ?>,
                                                slidesToScroll: 1
                                            }
                                        }
                                        // You can unslick at a given breakpoint now by adding:
                                        // settings: "unslick"
                                        // instead of a settings object
                                    ]
                                });
                            })
                        });
                    })(jQuery);
                </script>
<?php

            }


            return ob_get_clean();
        }


        /**
         * Throw error on object clone.
         *
         * The whole idea of the singleton design pattern is that there is a single
         * object therefore, we don't want the object to be cloned.
         *
         * @since 1.0
         * @access protected
         * @return void
         */
        public function __clone()
        {
            // Cloning instances of the class is forbidden.
            _doing_it_wrong(__FUNCTION__, esc_html('Cheatin&#8217; huh?', BDSC_TEXTDOMAIN), '1.0');
        }

        /**
         * Disable unserializing of the class.
         *
         * @since 1.0
         * @access protected
         * @return void
         */
        public function __wakeup()
        {
            // Unserializing instances of the class is forbidden.
            _doing_it_wrong(__FUNCTION__, esc_html('Cheatin&#8217; huh?', BDSC_TEXTDOMAIN), '1.0');
        }

        /**
         * It register the text domain to the WordPress
         */
        public function load_textdomain()
        {
            load_plugin_textdomain(BDSC_TEXTDOMAIN, false, BDSC_LANG_DIR);
        }

        public static function get_version_from_file_content( $file_path = '' ) {
            $version = '';

            if ( ! file_exists( $file_path ) ) {
                return $version;
            }

            $content = file_get_contents( $file_path );
            $version = self::get_version_from_content( $content );

            return $version;
        }

        public static function get_version_from_content( $content = '' ) {
            $version = '';

            if ( preg_match('/\*[\s\t]+?version:[\s\t]+?([0-9.]+)/i', $content, $v) ) {
                $version = $v[1];
            }

            return $version;
        }

        /**
         * Setup plugin constants.
         *
         * @access private
         * @since 1.0
         * @return void
         */
        private function setup_constants()
        {
            if ( ! defined( 'BDSC_FILE' ) ) {
                define( 'BDSC_FILE', __FILE__ );
            }

            $version = self::get_version_from_file_content( __FILE__ );

            require_once plugin_dir_path(__FILE__) . '/config.php'; // loads constant from a file so that it can be available on all files.

            // setup the updater
            if (!class_exists('EDD_SL_Plugin_Updater')) {
                // load our custom updater if it doesn't already exist
                include(dirname(__FILE__) . '/inc/EDD_SL_Plugin_Updater.php');
            }
            $license_key = trim(get_option('directorist_slider_license'));
            new EDD_SL_Plugin_Updater(ATBDP_AUTHOR_URL, __FILE__, array(
                'version' => BDSC_VERSION,        // current version number
                'license' => $license_key,    // license key (used get_option above to retrieve from DB)
                'item_id' => ATBDP_SLIDER_POST_ID,    // id of this plugin
                'author' => 'AazzTech',    // author of this plugin
                'url' => home_url(),
                'beta' => false // set to true if you wish customers to receive update notifications of beta releases
            ));
        }
    }

    if ( ! function_exists( 'directorist_is_plugin_active' ) ) {
        function directorist_is_plugin_active( $plugin ) {
            return in_array( $plugin, (array) get_option( 'active_plugins', array() ), true ) || directorist_is_plugin_active_for_network( $plugin );
        }
    }

    if ( ! function_exists( 'directorist_is_plugin_active_for_network' ) ) {
        function directorist_is_plugin_active_for_network( $plugin ) {
            if ( ! is_multisite() ) {
                return false;
            }

            $plugins = get_site_option( 'active_sitewide_plugins' );
            if ( isset( $plugins[ $plugin ] ) ) {
                    return true;
            }

            return false;
        }
    }

    /**
     * The main function for that returns BD_Slider_Carousel
     *
     * The main function responsible for returning the one true BD_Slider_Carousel
     * Instance to functions everywhere.
     *
     * Use this function like you would a global variable, except without needing
     * to declare the global.
     *
     *
     * @since 1.0
     * @return object|BD_Slider_Carousel The one true BD_Slider_Carousel Instance.
     */
    function BD_Slider_Carousel()
    {
        return BD_Slider_Carousel::instance();
    }

    if ( directorist_is_plugin_active( 'directorist/directorist-base.php' ) ) {
        BD_Slider_Carousel(); // get the plugin running
    }
}
